#include "dao.h"
#include "shard_settings_field_parser.h"

#include <solomon/libs/cpp/conf_db/model/shard_config.h>
#include <solomon/libs/cpp/ydb/dao_base.h>
#include <solomon/libs/cpp/ydb/paged_reader.h>
#include <solomon/libs/cpp/ydb/parser.h>
#include <solomon/libs/cpp/ydb/util.h>

#include <library/cpp/resource/resource.h>
#include <library/cpp/threading/future/future.h>

#include <util/generic/ptr.h>
#include <util/string/builder.h>
#include <util/string/vector.h>
#include <util/string/subst.h>

#include <utility>

using namespace NThreading;
using namespace NMonitoring;
using namespace NYdb::NTable;

namespace NSolomon::NDb {
namespace {

const TString CREATE_SHARD_TABLE = "shard_create";
const TString INSERT_SHARD = "insert_shard";
const TString DELETE_SHARD = "delete_shard";
const TString FIND_BY_ID = "shard_find_with_project";

class TShardParser: public TModelParser<NModel::TShardConfig> {
public:
    explicit TShardParser(const NYdb::TResultSet& rs)
        : TModelParser<NModel::TShardConfig>{rs}
    {
        using namespace NModel;

        FieldMappings_["id"] = Utf8FieldParser(Parser_, &TShardConfig::Id, "id");
        FieldMappings_["numId"] = IToUI32FieldParser(Parser_, &TShardConfig::NumId, "numId");
        FieldMappings_["clusterId"] = Utf8FieldParser(Parser_, &TShardConfig::ClusterId, "clusterId");
        FieldMappings_["clusterName"] = Utf8FieldParser(Parser_, &TShardConfig::ClusterName, "clusterName");
        FieldMappings_["createdAt"] = I64ToInstantFieldParser(Parser_, &TShardConfig::CreatedAt, "createdAt");
        FieldMappings_["createdBy"] = Utf8FieldParser(Parser_, &TShardConfig::CreatedBy, "createdBy");
        FieldMappings_["updatedAt"] = I64ToInstantFieldParser(Parser_, &TShardConfig::UpdatedAt, "updatedAt");
        FieldMappings_["updatedBy"] = Utf8FieldParser(Parser_, &TShardConfig::UpdatedBy, "updatedBy");
        FieldMappings_["maxFileSensors"] = IToUI32FieldParser(Parser_, &TShardConfig::MaxFileMetrics, "maxFileSensors");
        FieldMappings_["maxMemSensors"] = IToUI32FieldParser(Parser_, &TShardConfig::MaxMemMetrics, "maxMemSensors");
        FieldMappings_["maxResponseSizeBytes"] = IToUI32FieldParser(Parser_, &TShardConfig::MaxResponseSizeBytes, "maxResponseSizeBytes");
        FieldMappings_["maxSensorsPerUrl"] = IToUI32FieldParser(Parser_, &TShardConfig::MaxMetricsPerUrl, "maxSensorsPerUrl");
        FieldMappings_["numPartitions"] = IToUI32FieldParser(Parser_, &TShardConfig::NumPartitions, "numPartitions");
        FieldMappings_["projectId"] = Utf8FieldParser(Parser_, &TShardConfig::ProjectId, "projectId");
        FieldMappings_["folderId"] = Utf8FieldParser(Parser_, &TShardConfig::FolderId, "folderId");
        FieldMappings_["decimPolicy"] = Utf8FieldParser(Parser_, &TShardConfig::DecimPolicy, "decimPolicy");
        FieldMappings_["validationMode"] = Utf8FieldParser(Parser_, &TShardConfig::ValidationMode, "validationMode");
        FieldMappings_["sensorNameLabel"] = Utf8FieldParser(Parser_, &TShardConfig::MetricNameLabel, "sensorNameLabel");
        FieldMappings_["serviceId"] = Utf8FieldParser(Parser_, &TShardConfig::ServiceId, "serviceId");
        FieldMappings_["serviceName"] = Utf8FieldParser(Parser_, &TShardConfig::ServiceName, "serviceName");
        FieldMappings_["state"] = Utf8ToEnumFieldParser(Parser_, &TShardConfig::State, "state");
        FieldMappings_["version"] = IToUI32FieldParser(Parser_, &TShardConfig::Version, "version");
        FieldMappings_["description"] = Utf8FieldParser(Parser_, &TShardConfig::Description, "description");
        FieldMappings_["shardSettings"] = ShardSettingsFieldParser(Parser_, &TShardConfig::ShardSettings, "shardSettings");
        FieldMappings_["labels"] = Utf8FieldParser(Parser_, &TShardConfig::Labels, "labels");
    }
};


auto UnwrapOne = ParseOne<NModel::TShardConfig, TShardParser>;
auto UnwrapReadTableResult = ParseReadTableResult<NModel::TShardConfig, TShardParser>;

class TShardDao: public IShardConfigDao, private TDaoBase {
public:
    TShardDao(const TShardTables& tables, std::shared_ptr<TTableClient> client, TMetricRegistry& registry)
        : TDaoBase{std::move(client), registry}
        , TablePath_{tables.ShardTablePath}
        , NumPcsTablePath_{tables.NumPcsTablePath}
        , NumIdPcsTablePath_{tables.NumIdPcsTablePath}
    {
        Init();
    }

    const TString& Name() const override {
        static const TString NAME{"ShardsDao"};
        return NAME;
    }

    const TVector<TStringBuf>& Methods() const override {
        static const TVector<TStringBuf> METHODS{
            TStringBuf("createSchema"),
            TStringBuf("dropSchema"),
            TStringBuf("deleteByProject"),
            TStringBuf("insert"),
            TStringBuf("findAll"),
            TStringBuf("findById"),
        };

        return METHODS;
    }

    TAsyncVoid CreateTable() override {
        const auto req = RawQueries_.at(CREATE_SHARD_TABLE);

        return Execute<NYdb::TAsyncStatus>(*Client_, [req] (TSession session) {
            return session.ExecuteSchemeQuery(req);
        }).Apply(CompleteStatus(CounterFor("createSchema")));
    }

    TAsyncVoid DropTable() override {
        return Execute<NYdb::TAsyncStatus>(*Client_, [=] (TSession session) {
            return session.DropTable(TablePath_);
        }).Apply(CompleteStatus(CounterFor("dropSchema")));
    }


    TAsyncVoid Insert(const NModel::TShardConfig& model) override {
        auto req = RawQueries_.at(INSERT_SHARD);

        auto params = NYdb::TParamsBuilder{}
                .AddParam("$id").Utf8(model.Id).Build()
                .AddParam("$numId").Int32(model.NumId).Build()
                .AddParam("$clusterId").Utf8(model.ClusterId).Build()
                .AddParam("$clusterName").Utf8(model.ClusterName).Build()
                .AddParam("$serviceId").Utf8(model.ServiceId).Build()
                .AddParam("$serviceName").Utf8(model.ServiceName).Build()
                .AddParam("$projectId").Utf8(model.ProjectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$decimPolicy").Utf8(model.DecimPolicy).Build()
                .AddParam("$validationMode").Utf8(model.ValidationMode).Build()
                .AddParam("$createdAt").Int64(model.CreatedAt.MilliSeconds()).Build()
                .AddParam("$updatedAt").Int64(model.UpdatedAt.MilliSeconds()).Build()
                .AddParam("$createdBy").Utf8(model.CreatedBy).Build()
                .AddParam("$updatedBy").Utf8(model.UpdatedBy).Build()
                .AddParam("$version").Int32(model.Version).Build()
                .AddParam("$maxFileSensors").Int32(model.MaxFileMetrics).Build()
                .AddParam("$maxMemSensors").Int32(model.MaxMemMetrics).Build()
                .AddParam("$maxResponseSizeBytes").Int32(model.MaxResponseSizeBytes).Build()
                .AddParam("$maxSensorsPerUrl").Int32(model.MaxMetricsPerUrl).Build()
                .AddParam("$numPartitions").Int32(model.NumPartitions).Build()
                .AddParam("$state").Utf8(::ToString(model.State)).Build()
                .AddParam("$sensorNameLabel").Utf8(model.MetricNameLabel).Build()
                .AddParam("$description").Utf8(model.Description).Build()
                .AddParam("$shardSettings").Utf8(model.ShardSettings.ToJsonStr()).Build()
                .AddParam("$labels").Utf8(model.Labels).Build()
                // TODO: remove these stubs
                .AddParam("$sensorsTtlDays").Int32(0).Build()
            .Build();

        return Client_->RetryOperation<TDataQueryResult>([=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        }).Apply(CompleteStatus(CounterFor("insert")));
    }

    TAsyncVoid Delete(const TString& shardId, const TString& projectId) override {
        const auto req = RawQueries_.at(DELETE_SHARD);
        auto params = NYdb::TParamsBuilder{}
                .AddParam("$id").Utf8(shardId).Build()
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
            .Build();

        auto f = [=] (NYdb::NTable::TSession s) {
            return ExecutePrepared(std::move(s), req, params);
        };

        return Client_->RetryOperation<TDataQueryResult>(f)
            .Apply(CompleteStatus(CounterFor("deleteByProject")));
    }

    TAsyncShardConfigs GetAll() override {
        return ReadTable<NModel::TShardConfig>(*Client_, TablePath_, UnwrapReadTableResult, CounterFor("findAll"));
    }

    TAsyncShardConfig GetById(const TString& shardId, const TString& projectId) override {
        auto q = RawQueries_.at(FIND_BY_ID);
        auto params = NYdb::TParamsBuilder{}
                .AddParam("$projectId").Utf8(projectId).Build()
                .AddParam("$folderId").Utf8("").Build()
                .AddParam("$id").Utf8(shardId).Build()
            .Build();

        auto p = NewPromise<std::optional<NModel::TShardConfig>>();
        ReadDataWithRetries(*Client_, q, params, p, UnwrapOne, CounterFor("findById"));
        return p.GetFuture();
    }

private:
    THashMap<TStringBuf, TString> LoadQueries() override {
        THashMap<TStringBuf, TString> result;

        auto substTables = [&] (TString& query) {
            SubstGlobal(query, TStringBuf("${shard.table.path}"), TablePath_);
            SubstGlobal(query, TStringBuf("${shard.pcs.key.table.path}"), NumPcsTablePath_);
            SubstGlobal(query, TStringBuf("${shard.pcs.numId.table.path}"), NumIdPcsTablePath_);
        };

        auto getOne = [&] (TStringBuf key, TStringBuf resourceKey) {
            auto resource = NResource::Find(resourceKey);
            Y_ENSURE(!resource.empty(), "Resource " << resourceKey << " is empty");
            substTables(resource);
            result.emplace(key, resource);
        };

        const auto resource = NResource::Find("shard_create_table.yql");

        // XXX: change the query itself
        TVector<TStringBuf> lines;
        TStringBuilder schemeQueryBuilder;

        bool isSchemePart = true;
        for (auto tok: StringSplitter(resource).Split('\n')) {
            const auto line = tok.Token();

            if (line.StartsWith("COMMIT;")) {
                isSchemePart = false;
                continue;
            }

            if (isSchemePart) {
                schemeQueryBuilder << line << "\n";
            }
        }

        auto& schemeQuery = result[CREATE_SHARD_TABLE];
        schemeQuery = schemeQueryBuilder;
        substTables(schemeQuery);

        getOne(INSERT_SHARD, "shard_insert.yql");
        getOne(DELETE_SHARD, "shard_delete_with_project.yql");
        getOne(FIND_BY_ID, "shard_find_with_project.yql");

        return result;
    }

private:
    const TString TablePath_;
    const TString NumPcsTablePath_;
    const TString NumIdPcsTablePath_;
};

} // namespace

IShardConfigDaoPtr CreateYdbShardDao(const TShardTables& tables, std::shared_ptr<TTableClient> client, TMetricRegistry& registry) {
    return ::MakeIntrusive<TShardDao>(tables, std::move(client), registry);
}

} // namespace NSolomon::NDb
