#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/conf_db/model/cluster_config.h>
#include <solomon/libs/cpp/conf_db/model/shard_config.h>
#include <solomon/libs/cpp/conf_db/model/service_config.h>
#include <solomon/libs/cpp/conf_db/model/agent_config.h>
#include <solomon/libs/cpp/conf_db/ydb/ut_helpers/config_creators.h>
#include <solomon/libs/cpp/ydb/config/ydb_config.pb.h>

#include <ydb/public/sdk/cpp/client/ydb_table/table.h>

#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/testing/common/network.h>
#include <library/cpp/testing/gtest/gtest.h>

#include <util/stream/file.h>

using namespace NSolomon::NDb;
using namespace NSolomon::NTesting;
using namespace NSolomon::NDb::NModel;
using namespace NMonitoring;

constexpr auto CLUSTER_TABLE_NAME = "/local/Clusters";
constexpr auto SERVICE_TABLE_NAME = "/local/Services";
constexpr auto SHARD_TABLE_NAME = "/local/Shards";
constexpr auto AGENT_TABLE_NAME = "/local/Agent";
constexpr auto PROJECT_TABLE_NAME = "/local/Projects";
constexpr auto PROVIDER_TABLE_NAME = "/local/Providers";

namespace NYdb {
    std::ostream& operator<<(std::ostream& os, const TColumn& column) {
        TString s = column.Name + " " + FormatType(column.Type);
        os.write(s.data(), s.size());
        return os;
    }
}

namespace NSolomon::NDb::NModel {
    template <typename T>
    std::ostream& operator<<(std::ostream& os, const T& model) {
        TString s = ToString(model) + '\n';
        os.write(s.data(), s.size());
        return os;
    }

    template <typename T>
    std::ostream& operator<<(std::ostream& os, const std::optional<T>& maybe) {
        if (!maybe) {
            os << TStringBuf("null");
            return os;
        }

        return operator<<(os, *maybe);
    }
}

class TYdbTest: public testing::Test {
public:
    void SetUp() override {
        if (!Conn_) {
            TYdbConfig config;
            config.SetAddress(TFileInput{"ydb_endpoint.txt"}.ReadLine());
            config.SetDatabase(TFileInput{"ydb_database.txt"}.ReadLine());
            config.SetPagingLimit(1);
            Conn_ = CreateYdbConnection(config, Registry_);
        }

        TShardTables shardTables {
            .ShardTablePath = SHARD_TABLE_NAME,
            .NumPcsTablePath = "/local/ShardPcsKey",
            .NumIdPcsTablePath = "/local/ShardPcsNumId",
        };

        ProjectDao_ = Conn_->CreateProjectDao(PROJECT_TABLE_NAME);
        ClusterDao_ = Conn_->CreateClusterDao(CLUSTER_TABLE_NAME);
        ServiceDao_ = Conn_->CreateServiceDao(SERVICE_TABLE_NAME);
        ShardDao_ = Conn_->CreateShardDao(shardTables);
        AgentDao_ = Conn_->CreateAgentDao(AGENT_TABLE_NAME);
        ProviderDao_ = Conn_->CreateProviderDao(PROVIDER_TABLE_NAME);

        ProjectDao_->CreateTable().GetValueSync();
        ClusterDao_->CreateTable().GetValueSync();
        ServiceDao_->CreateTable().GetValueSync();
        ShardDao_->CreateTable().GetValueSync();
        ProviderDao_->CreateTable().GetValueSync();
        AgentDao_->CreateTable().GetValueSync();
    }

    void TearDown() override {
        auto cluster = MakeClusterConfig();
        ClusterDao_->Delete(cluster.Id, cluster.ProjectId).GetValueSync();

        auto shard = MakeShardConfig();
        ShardDao_->Delete(shard.Id, shard.ProjectId).GetValueSync();

        auto service = MakeServiceConfig();
        ServiceDao_->Delete(service.Id, service.ProjectId).GetValueSync();

        auto project = MakeProjectConfig();
        ProjectDao_->Delete(project.Id).GetValueSync();

        auto provider = MakeProviderConfig();
        ProviderDao_->Delete(provider.Id).GetValueSync();

        auto agent = MakeAgentConfig();
        AgentDao_->DeleteByProvider(agent.Provider).GetValueSync();
        AgentDao_->DeleteByProvider("differentProvider").GetValueSync();
    }

protected:
    TMetricRegistry Registry_;
    NSolomon::NDb::IDbConnectionPtr Conn_;
    NSolomon::NDb::IClusterConfigDaoPtr ClusterDao_;
    NSolomon::NDb::IShardConfigDaoPtr ShardDao_;
    NSolomon::NDb::IServiceConfigDaoPtr ServiceDao_;
    NSolomon::NDb::IAgentConfigDaoPtr AgentDao_;
    NSolomon::NDb::IProjectConfigDaoPtr ProjectDao_;
    NSolomon::NDb::IProviderConfigDaoPtr ProviderDao_;
};

TEST_F(TYdbTest, ClusterConfigListByProject) {
    auto expect = MakeClusterConfig();
    ClusterDao_->Insert(expect).GetValueSync();

    auto val = ClusterDao_->GetByProject(expect.Id, expect.ProjectId)
            .ExtractValueSync();

    ASSERT_EQ(*val, MakeClusterConfig());

    val = ClusterDao_->GetByProject("other-cluster", "test-project")
            .ExtractValueSync();

    ASSERT_FALSE(val.has_value());
}

TEST_F(TYdbTest, ServiceConfigListByProject) {
    auto expect = MakeServiceConfig();
    ServiceDao_->Insert(expect).GetValueSync();
    auto val = ServiceDao_->GetByProject(expect.Id, expect.ProjectId)
            .ExtractValueSync();

    ASSERT_EQ(*val, MakeServiceConfig());

    val = ServiceDao_->GetByProject("other-service", "test-project")
            .ExtractValueSync();

    ASSERT_FALSE(val.has_value());
}

TEST_F(TYdbTest, ProjectInsertDelete) {
    auto project = MakeProjectConfig();
    ProjectDao_->Insert(project).GetValueSync();
    auto projects = ProjectDao_->GetAll().ExtractValueSync();
    ASSERT_THAT(projects, testing::ElementsAre(project));

    ProjectDao_->Delete(project.Id).GetValueSync();
    projects = ProjectDao_->GetAll().ExtractValueSync();
    ASSERT_TRUE(projects.empty());
}

TEST_F(TYdbTest, ProjectExists) {
    auto project = MakeProjectConfig();
    ProjectDao_->Insert(project).GetValueSync();
    auto exists = ProjectDao_->Exists(project.Id).ExtractValueSync();
    ASSERT_TRUE(exists);
    exists = ProjectDao_->Exists("foo").ExtractValueSync();
    ASSERT_FALSE(exists);
}

TEST_F(TYdbTest, ProjectFindById) {
    auto project = MakeProjectConfig();
    ProjectDao_->Insert(project).GetValueSync();
    auto actual = ProjectDao_->GetById(project.Id).ExtractValueSync();
    ASSERT_EQ(actual, project);

    auto nonExisting = ProjectDao_->GetById("foo").ExtractValueSync();
    ASSERT_FALSE(nonExisting.has_value());
}

TEST_F(TYdbTest, ProviderInsertDelete) {
    auto provider = MakeProviderConfig();
    ProviderDao_->Insert(provider).GetValueSync();
    auto providers = ProviderDao_->GetAll().ExtractValueSync();
    ASSERT_THAT(providers, testing::ElementsAre(provider));

    ProviderDao_->Delete(provider.Id).GetValueSync();
    providers = ProviderDao_->GetAll().ExtractValueSync();
    ASSERT_TRUE(providers.empty());
}

TEST_F(TYdbTest, ProviderExists) {
    auto provider = MakeProviderConfig();
    ProviderDao_->Insert(provider).GetValueSync();
    auto exists = ProviderDao_->Exists(provider.Id).ExtractValueSync();
    ASSERT_TRUE(exists);
    exists = ProviderDao_->Exists("foo").ExtractValueSync();
    ASSERT_FALSE(exists);
}

TEST_F(TYdbTest, ProviderFindById) {
    auto provider = MakeProviderConfig();
    ProviderDao_->Insert(provider).GetValueSync();
    auto actual = ProviderDao_->GetById(provider.Id).ExtractValueSync();
    ASSERT_EQ(actual, provider);

    auto nonExisting = ProviderDao_->GetById("foo").ExtractValueSync();
    ASSERT_FALSE(nonExisting.has_value());
}

TEST_F(TYdbTest, ProviderReference) {
    auto provider = MakeProviderConfig();
    {
        TReferenceConf reference;
        reference.Label = "resource_id";
        reference.Services.push_back("compute");
        reference.Types.push_back("vm");
        reference.Types.push_back("disk");
        provider.References.push_back(reference);
    }
    ProviderDao_->Insert(provider).GetValueSync();
    auto actual = ProviderDao_->GetById(provider.Id).ExtractValueSync();
    ASSERT_EQ(actual->References, provider.References);
}

TEST_F(TYdbTest, ThrowsOnPkViolation) {
    {
        auto model = MakeClusterConfig();
        ClusterDao_->Insert(model).GetValueSync();
        ASSERT_THROW(ClusterDao_->Insert(model).GetValueSync(), yexception);
    }

    {
        auto model = MakeServiceConfig();
        ServiceDao_->Insert(model).GetValueSync();
        ASSERT_THROW(ServiceDao_->Insert(model).GetValueSync(), yexception);
    }

    {
        auto model = MakeShardConfig();
        ShardDao_->Insert(model).GetValueSync();
        ASSERT_THROW(ShardDao_->Insert(model).GetValueSync(), yexception);
    }
}

TEST_F(TYdbTest, ThrowsOnWrongAddress) {
    auto portHolder = ::NTesting::GetFreePort();
    TYdbConfig config;
    config.SetAddress("localhost:" + ToString<ui16>(portHolder));

    NMonitoring::TMetricRegistry registry;
    auto conn = CreateYdbConnection(config, registry);

    auto mapper = conn->CreateClusterDao(CLUSTER_TABLE_NAME);
    ASSERT_THROW(mapper->CreateTable().GetValueSync(), yexception);
}

TEST_F(TYdbTest, ShardConfigList) {
    ClusterDao_->Insert(MakeClusterConfig()).GetValueSync();
    ServiceDao_->Insert(MakeServiceConfig()).GetValueSync();

    auto expected = MakeShardConfig();
    ShardDao_->Insert(expected).GetValueSync();

    auto list = ShardDao_->GetAll().ExtractValueSync();

    ASSERT_THAT(list, testing::ElementsAre(expected));
}

TEST_F(TYdbTest, PagedReaderTest) {
    ClusterDao_->Insert(MakeClusterConfig()).GetValueSync();
    ServiceDao_->Insert(MakeServiceConfig()).GetValueSync();

    const auto count = 10;
    TVector<TShardConfig> shards;

    for (auto i = 0; i < count; ++i) {
        auto shard = MakeShardConfig();
        auto idxStr = ToString(i);
        shard.Id = "id" + idxStr;
        shard.ClusterName = "clusterName" + idxStr;
        shards.push_back(shard);
        ShardDao_->Insert(shard).GetValueSync();
    }

    auto result = ShardDao_->GetAll().GetValueSync();
    ASSERT_THAT(result, testing::ElementsAreArray(shards));
}

TEST_F(TYdbTest, AgentDeletion) {
    auto toDelete = MakeAgentConfig();
    AgentDao_->InsertOrUpdate(toDelete).GetValueSync();
    AgentDao_->Delete(toDelete.Hostname, toDelete.Provider).GetValueSync();

    auto agents = AgentDao_->GetAll().ExtractValueSync();
    ASSERT_TRUE(agents.empty());
}

TEST_F(TYdbTest, AgentByProjectDeletion) {
    auto first = MakeAgentConfig();
    first.Hostname = "another.host";
    AgentDao_->InsertOrUpdate(first).GetValueSync();

    auto second = MakeAgentConfig();
    second.Hostname = "another.host2";
    second.Provider = "differentProvider";
    AgentDao_->InsertOrUpdate(second).GetValueSync();

    AgentDao_->DeleteByProvider("provider").GetValueSync();

    auto agents = AgentDao_->GetAll().ExtractValueSync();
    ASSERT_THAT(agents, testing::ElementsAre(second));
}

TEST_F(TYdbTest, AgentUpdate) {
    auto agent = MakeAgentConfig();
    agent.LastSeen = TInstant::Seconds(100500);
    agent.Description = "this is a description";
    agent.DataPort++;
    agent.ManagementPort++;

    AgentDao_->InsertOrUpdate(agent).GetValueSync();
    auto agents = AgentDao_->GetAll().ExtractValueSync();
    ASSERT_THAT(agents, testing::ElementsAre(agent));
}

TEST_F(TYdbTest, MetricConfParseCheck) {
    TMetricConf metricConf = TMetricConf::FromString(METRIC_CONF_TEST_DATA);
    ASSERT_TRUE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 3u);

    for (int i = 0; i < 3; ++i) {
        ASSERT_EQ(metricConf.AggrRules[i].Cond.size(), 1u);
        ASSERT_EQ(metricConf.AggrRules[i].Target.size(), 1u);
    }

    ASSERT_EQ(metricConf.AggrRules[0].Cond[0], "dc=sas");
    ASSERT_EQ(metricConf.AggrRules[0].Target[0], "host=cluster_sas");

    ASSERT_EQ(metricConf.AggrRules[1].Cond[0], "dc=man");
    ASSERT_EQ(metricConf.AggrRules[1].Target[0], "host=cluster_man");

    ASSERT_EQ(metricConf.AggrRules[2].Cond[0], "host=*");
    ASSERT_EQ(metricConf.AggrRules[2].Target[0], "host=cluster");
}

TEST_F(TYdbTest, MetricConfParseEmptyCheck) {
    TMetricConf metricConf = TMetricConf::FromString("");
    ASSERT_FALSE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}

TEST_F(TYdbTest, MetricConfParseEmptyJsonCheck) {
    TMetricConf metricConf = TMetricConf::FromString(R"({})");
    ASSERT_FALSE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}

TEST_F(TYdbTest, MetricConfParseInvalidJsonCheck) {
    TMetricConf metricConf = TMetricConf::FromString(R"({)");
    ASSERT_FALSE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}

TEST_F(TYdbTest, MetricConfParseIncompleteWithMemCheck) {
    TMetricConf metricConf = TMetricConf::FromString(R"({"rawDataMemOnly": true})");
    ASSERT_TRUE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}

TEST_F(TYdbTest, MetricConfParseIncompleteWithoutMemCheck) {
    TMetricConf metricConf = TMetricConf::FromString(R"({"aggrRules": []})");
    ASSERT_FALSE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}

TEST_F(TYdbTest, MetricConfParseBadAggrRulesTypeCheck) {
    TMetricConf metricConf = TMetricConf::FromString(R"({"aggrRules": false})");
    ASSERT_FALSE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}

TEST_F(TYdbTest, MetricConfParseBadRawDataMemOnlyTypeCheck) {
    TMetricConf metricConf = TMetricConf::FromString(R"({"rawDataMemOnly": []})");
    ASSERT_FALSE(metricConf.RawDataMemOnly);
    ASSERT_EQ(metricConf.AggrRules.size(), 0u);
}
