#include "dao.h"

#include <solomon/libs/cpp/conf_db/db.h>
#include <solomon/libs/cpp/conf_db/model/cluster_config.h>
#include <solomon/libs/cpp/secrets/secrets.h>
#include <solomon/libs/cpp/ydb/config/ydb_config.pb.h>
#include <solomon/libs/cpp/ydb/driver.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_result/result.h>
#include <ydb/public/sdk/cpp/client/ydb_table/table.h>
#include <kikimr/public/sdk/cpp/client/tvm/tvm.h>

#include <library/cpp/threading/future/future.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/string/subst.h>

namespace NSolomon::NDb {
namespace {

using namespace NThreading;
using namespace NMonitoring;
using NYdb::NTable::TTableClient;
using NYdb::NTable::TTxControl;

static std::function<i64()> SafelyGet(const std::weak_ptr<TTableClient>& client, i64 (TTableClient::*getter)() const) {
    return [=]() {
        i64 count = 0;
        if (auto p = client.lock()) {
            count = std::invoke(getter, *p);
        }
        return count;
    };
}

class TYdbConnection: public IDbConnection {
public:
    TYdbConnection(const NYdb::TDriver& driver, TMetricRegistry& registry, const NYdb::NTable::TClientSettings& config = NYdb::NTable::TClientSettings())
    : Client_{std::make_shared<TTableClient>(driver, std::move(config))}
        , Registry_{registry}

    {
        Registry_.LazyIntGauge(
            {{"sensor", "ydb.sessionPool.active"}},
            SafelyGet(Client_, &TTableClient::GetActiveSessionCount));
        Registry_.LazyIntGauge(
            {{"sensor", "ydb.sessionPool.limit"}},
            SafelyGet(Client_, &TTableClient::GetActiveSessionsLimit));
        Registry_.LazyIntGauge(
            {{"sensor", "ydb.sessionPool.currentSize"}},
            SafelyGet(Client_, &TTableClient::GetCurrentPoolSize));
    }

    IClusterConfigDaoPtr CreateClusterDao(const TString& path) override {
        return CreateYdbClusterDao(path, Client_, Registry_);
    }

    IShardConfigDaoPtr CreateShardDao(const TShardTables& tables) override {
        return CreateYdbShardDao(tables, Client_, Registry_);
    }

    IServiceConfigDaoPtr CreateServiceDao(const TString& path) override {
        return CreateYdbServiceDao(path, Client_, Registry_);
    }

    IAgentConfigDaoPtr CreateAgentDao(const TString& path) override {
        return CreateYdbAgentDao(path, Client_, Registry_);
    }

    IProjectConfigDaoPtr CreateProjectDao(const TString& path) override {
        return CreateYdbProjectDao(path, Client_, Registry_);
    }

    IProviderConfigDaoPtr CreateProviderDao(const TString& path) override {
        return CreateYdbProviderDao(path, Client_, Registry_);
    }

private:
    std::shared_ptr<TTableClient> Client_;
    TMetricRegistry& Registry_;
};

} // namespace

// NOLINTNEXTLINE(performance-unnecessary-value-param): false positive
IDbConnectionPtr CreateYdbConnection(NYdb::TDriver driver, TMetricRegistry& registry) {
    return ::MakeIntrusive<TYdbConnection>(std::move(driver), registry);
}

IDbConnectionPtr CreateYdbConnection(const NYdb::TDriver& driver, const TSessionPoolConfig& poolConfig, TMetricRegistry& registry) {
    NYdb::NTable::TClientSettings settings;

    if (auto maxActiveSessions = poolConfig.GetMaxActiveSessions()) {
        settings.SessionPoolSettings_.MaxActiveSessions(maxActiveSessions);
    }
    if (auto minPoolSize = poolConfig.GetMinPoolSize()) {
        settings.SessionPoolSettings_.MinPoolSize(minPoolSize);
    }
    return ::MakeIntrusive<TYdbConnection>(driver, registry, settings);
}

IDbConnectionPtr CreateYdbConnection(const TYdbConfig& config, TMetricRegistry& registry) {
    TSessionPoolConfig poolConfig = config.GetSessionPoolConfig();
    NYdb::NTable::TClientSettings settings;

    if (auto maxActiveSessions = poolConfig.GetMaxActiveSessions()) {
        settings.SessionPoolSettings_.MaxActiveSessions(maxActiveSessions);
    }
    if (auto minPoolSize = poolConfig.GetMinPoolSize()) {
        settings.SessionPoolSettings_.MinPoolSize(minPoolSize);
    }
    auto secretProvider = NSecrets::EmptySecretProvider();
    return ::MakeIntrusive<TYdbConnection>(CreateDriver(config, *secretProvider), registry, settings);
}

} // namespace NSolomon::NDb
