#include "leader.h"

#include <solomon/libs/cpp/actors/events/events.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/events.h>

#include <utility>

using namespace NActors;

namespace NSolomon::NCoordination {
namespace {
    enum class EState {
        Undefined,
        Follower,
        Leader,
        Defunct,
    };

    struct TLeaderEvents: private TPrivateEvents {
        enum {
            EvLock = SpaceBegin,
            EvUnlock,
            EvChanged,
            EvError,
            EvConnect,
            End,
        };
        static_assert(End < SpaceEnd, "too many event types");
    };

    struct TEvLock: TEventLocal<TEvLock, TLeaderEvents::EvLock> {
    };

    struct TEvUnlock: TEventLocal<TEvUnlock, TLeaderEvents::EvUnlock> {
    };

    struct TEvConnect: TEventLocal<TEvConnect, TLeaderEvents::EvConnect> {
    };

    struct TEvError: TEventLocal<TEvError, TLeaderEvents::EvError> {
        TEvError(TString s)
            : Message{std::move(s)}
        {
        }

        TString Message;
    };

    struct TEvChanged: TEventLocal<TEvChanged, TLeaderEvents::EvChanged> {
        TEvChanged(TString leaderId, ui64 orderId)
            : LeaderId{std::move(leaderId)}
            , OrderId{orderId}
        {
        }

        TString LeaderId;
        ui64 OrderId{0};
    };

    class TLockStateListener: public ILockStateListener {
    public:
        TLockStateListener(TActorSystem* actorSystem, NActors::TActorId replyTo)
            : ActorSystem_{actorSystem}
            , ReplyTo_{replyTo}
        {}

    private:
        // ILockStateListener
        void OnLock() override {
            ActorSystem_->Send(ReplyTo_, new TEvLock);
        }

        void OnUnlock() override {
            ActorSystem_->Send(ReplyTo_, new TEvUnlock);
        }

        void OnChanged(const TLockDescription& description) override {
            ActorSystem_->Send(ReplyTo_, new TEvChanged{description.Data, description.OrderId});
        }

        void OnError(TString message) override {
            ActorSystem_->Send(ReplyTo_, new TEvError{std::move(message)});
        }

    private:
        TActorSystem* ActorSystem_;
        NActors::TActorId ReplyTo_;
    };

    class TClusterMemberActor
        : public TActorBootstrapped<TClusterMemberActor>
    {
    public:
        TClusterMemberActor(IDistributedLockPtr lock, THolder<IClusterMember> impl, TDuration reconnectTimeout)
            : Lock_{std::move(lock)}
            , Impl_{std::move(impl)}
            , ReconnectTimeout_{reconnectTimeout}
        {
        }

        void Bootstrap() {
            ActorSystem_ = TActorContext::ActorSystem();
            SelfId_ = SelfId();

            LockStateListener_ = MakeIntrusive<TLockStateListener>(ActorSystem_, SelfId_);

            Become(&TThis::StateInit);
            TryAcquireLock();
        }

        STATEFN(StateDefunct) {
            switch (ev->GetTypeRewrite()) {
                cFunc(TEvents::TEvWakeup::EventType, OnWakeup);
                cFunc(TEvents::TSystem::PoisonPill, PassAway);
            }
        }

        STATEFN(StateInit) {
            switch (ev->GetTypeRewrite()) {
                cFunc(TEvConnect::EventType, OnConnect);
                cFunc(TEvLock::EventType, OnLockAcquired);
                hFunc(TEvError, OnError);
                hFunc(TEvChanged, OnChanged);
                cFunc(TEvents::TSystem::PoisonPill, PassAway);
            }
        }

        STATEFN(StateCandidate) {
            switch (ev->GetTypeRewrite()) {
                cFunc(TEvents::TSystem::PoisonPill, ReleaseLock);
                cFunc(TEvLock::EventType, OnLockAcquired);
                cFunc(TEvUnlock::EventType, OnLockReleased);
                hFunc(TEvError, OnError);
                hFunc(TEvChanged, OnChanged);
            }
        }

        STATEFN(StateLeader) {
            switch (ev->GetTypeRewrite()) {
                cFunc(TEvents::TSystem::PoisonPill, ReleaseLock);
                cFunc(TEvLock::EventType, OnLockAcquired);
                cFunc(TEvUnlock::EventType, OnLockReleased);
                hFunc(TEvError, OnError);
                hFunc(TEvChanged, OnChanged);
            }
        }

        STATEFN(StateDying) {
            switch (ev->GetTypeRewrite()) {
                cFunc(TEvents::TSystem::PoisonPill, PassAway);
            }
        }

        void OnError(const TEvError::TPtr& ev) {
            BecomeDefunct(ev->Get()->Message);
        }

        void OnConnect() {
            BecomeCandidate();
        }

        void OnLockAcquired() {
            BecomeLeader();
        }

        void OnLockReleased() {
            BecomeCandidate();
        }

        void OnWakeup() {
            Become(&TThis::StateInit);
            TryAcquireLock();
        }

        void OnChanged(const TEvChanged::TPtr& ev) {
            Impl_->OnLeaderChanged(std::move(ev->Get()->LeaderId), ev->Get()->OrderId);
        }

        void TryAcquireLock() {
            Lock_->Acquire(LockStateListener_).Subscribe([as{ActorSystem_}, replyTo{SelfId_}] (auto) {
                as->Send(replyTo, new TEvConnect);
            });
        }

    private:
        void ReleaseLock() {
            Lock_->Release().Subscribe([=] (auto) {
                ActorSystem_->Send(SelfId_, new TEvents::TEvPoisonPill);
            });

            Become(&TThis::StateDying);
        }

        void BecomeLeader() {
            Become(&TThis::StateLeader);
            Y_VERIFY_DEBUG(State_ != EState::Leader);
            if (State_ != EState::Leader) {
                Impl_->OnBecomeLeader();
            }

            State_ = EState::Leader;
        }

        void BecomeCandidate() {
            Become(&TThis::StateCandidate);
            if (State_ != EState::Follower) {
                Impl_->OnBecomeFollower();
            }

            State_ = EState::Follower;
        }

        void BecomeDefunct(const TString& message) {
            Become(&TThis::StateDefunct);
            Schedule(ReconnectTimeout_, new TEvents::TEvWakeup);

            Impl_->OnError(message);
            State_ = EState::Defunct;
        }

    private:
        IDistributedLockPtr Lock_;
        TActorSystem* ActorSystem_{nullptr};
        TActorId SelfId_;
        THolder<IClusterMember> Impl_;
        EState State_{EState::Undefined};
        TDuration ReconnectTimeout_;

        ILockStateListenerPtr LockStateListener_;
    };
} // namespace
    IActor* CreateClusterMemberActor(IDistributedLockPtr lock, THolder<IClusterMember> impl, TDuration reconnectTimeout) {
        return new TClusterMemberActor{std::move(lock), std::move(impl), reconnectTimeout};
    }
} // namespace NSolomon::NCoordination
