#include <solomon/libs/cpp/coordination/testlib/lock.h>
#include <solomon/libs/cpp/coordination/testlib/actor_runtime.h>
#include <solomon/libs/cpp/coordination/leader_election/leader.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <library/cpp/actors/interconnect/interconnect.h>
#include <library/cpp/actors/interconnect/interconnect_tcp_server.h>
#include <library/cpp/actors/testlib/test_runtime.h>

#include <util/stream/file.h>
#include <util/string/strip.h>

using namespace testing;
using namespace NActors;
using namespace NThreading;
using namespace NSolomon::NCoordination;
using namespace NSolomon::NTesting;
using namespace NSolomon;
using namespace NYdb;
using namespace NYdb::NCoordination;


struct TYdb {
public:
    TYdb()
        : Database{Strip(TFileInput("ydb_database.txt").ReadAll())}
        , Driver{
            TDriverConfig{}
                .SetEndpoint(Strip(TFileInput("ydb_endpoint.txt").ReadAll()))
                .SetDatabase(Database)
        }
        , Client{Driver}
        , NodePath_{TStringBuilder() << "/" << Database << "/foo"}
        , LockName_{CreateGuidAsString()}
    {
    }

    void Init() {
        auto s = Client.CreateNode(NodePath_, {}).GetValueSync();
        if (!s.IsSuccess()) {
            Y_FAIL();
        }

        auto r = Client.StartSession(NodePath_).ExtractValueSync();
        auto session = r.GetResult();
        auto createResult = session.CreateSemaphore(LockName_, 1).GetValueSync();
    }

    IDistributedLockPtr MakeLock(ui32 nodeId) {
        return CreateYdbLock(TYdbLockConfig{
            .Driver = Driver,
            .Path = NodePath_,
            .Name = LockName_,
            .Data = ToString(nodeId),
        });
    }

    TString Database;
    TDriver Driver;
    TClient Client;

private:
    TString NodePath_;
    TString LockName_;
};

struct TClusterMember: IClusterMember {
    enum EState {
        ES_UNDEFINED,
        ES_LEADER,
        ES_FOLLOWER,
        ES_ERROR,
    };

    void OnBecomeLeader() override {
        Y_VERIFY(State != ES_LEADER);
        State = ES_LEADER;
        LeaderEv_.Signal();
    }

    void OnBecomeFollower() override {
        Y_VERIFY(State != ES_FOLLOWER);
        State = ES_FOLLOWER;
        FollowerEv_.Signal();
    }

    void OnLeaderChanged(TString leaderId, ui64 orderId) override {
        CurrentLeader = leaderId;
        OrderId = orderId;
        ChangedEv_.Signal();
    }

    void OnError(const TString&) override {
        State = ES_ERROR;
        ErrorEv_.Signal();
    }

    void WaitLeader() {
        LeaderEv_.WaitI();
    }

    void WaitFollower() {
        FollowerEv_.WaitI();
    }

    void WaitChanged() {
        ChangedEv_.WaitI();
    }

    void WaitError() {
        ErrorEv_.WaitI();
    }

    EState State{ES_UNDEFINED};
    ui64 OrderId{0};
    TString CurrentLeader;

private:
    TAutoEvent LeaderEv_;
    TAutoEvent FollowerEv_;
    TAutoEvent ChangedEv_;
    TAutoEvent ErrorEv_;
};

class TLeaderElectionFixture: public ::testing::Test {
protected:
    void SetUp() override {
        ActorSystem_.Reset(new TCoordinationActorRuntime{1, false});
        ActorSystem_->Initialize();
        Lock_ = {};

        auto obj = MakeHolder<TClusterMember>();
        ClusterMember_ = obj.Get();
        Idx_ = 0;

        auto* actor = CreateClusterMemberActor(
            MakeLock(),
            std::move(obj)
        );

        auto aid = ActorSystem_->Register(
            actor
        );

        ActorSystem_->EnableScheduleForActor(aid, true);
        ActorSystem_->WaitForBootstrap();
    }

    void TearDown() override {
    }

protected:
    virtual IDistributedLockPtr MakeLock() = 0;

    THolder<TCoordinationActorRuntime> ActorSystem_;
    TMockLock Lock_;
    ui32 Idx_{0};
    TClusterMember* ClusterMember_{};
};

class TLeaderElectionSlowInitTest: public TLeaderElectionFixture {
protected:
    IDistributedLockPtr MakeLock() override {
        return Lock_.CreateSlowLockClient(Idx_, TDuration::MilliSeconds(200), TDuration::MilliSeconds(200));
    }
};

TEST_F(TLeaderElectionSlowInitTest, ErrorOnInitProducesCallback) {
    Lock_.ErrorFor(Idx_);

    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_ERROR));
}

class TLeaderElectionTest: public TLeaderElectionFixture {
protected:
    IDistributedLockPtr MakeLock() override {
        auto lock = std::make_shared<TLockCallProxy>(
            Lock_.CreateLockClient(Idx_)
        );

        LockProxy_ = lock;
        return lock;
    }

    std::shared_ptr<TLockCallProxy> LockProxy_;
};

TEST_F(TLeaderElectionTest, ErrorWhileCandidate) {
    Lock_.AcquireFor(Idx_);

    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_LEADER));

    Lock_.ReleaseFor(Idx_);
    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_FOLLOWER));

    Lock_.ErrorFor(Idx_);
    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_ERROR));
}

TEST_F(TLeaderElectionTest, ErrorWhileLeader) {
    // not sure if error callback can be called before the release in a real setting, but anyway
    Lock_.AcquireFor(Idx_);
    Lock_.ErrorFor(Idx_);

    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_ERROR));
}

TEST_F(TLeaderElectionTest, LockDescriptionIsCorrectSingle) {
    Lock_.AcquireFor(Idx_);

    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_LEADER));
    ASSERT_THAT(ClusterMember_->OrderId, Lock_.OrderId());
    ASSERT_THAT(ClusterMember_->CurrentLeader, ToString(*Lock_.CurrentHolder()));
}

TEST_F(TLeaderElectionTest, LockDescriptionIsCorrect) {
    TClusterMember* second{nullptr};
    ui32 secondIdx{1};
    // add one more member to the cluster
    {
        auto obj = MakeHolder<TClusterMember>();
        second = obj.Get();

        ActorSystem_->Register(CreateClusterMemberActor(
            Lock_.CreateLockClient(secondIdx),
            std::move(obj)
        ));

        ActorSystem_->WaitForBootstrap();
    }

    Lock_.AcquireFor(secondIdx);
    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_FOLLOWER));
    ASSERT_THAT(ClusterMember_->OrderId, Lock_.OrderId());
    ASSERT_THAT(ClusterMember_->CurrentLeader, ToString(*Lock_.CurrentHolder()));
    ASSERT_THAT(second->State, Eq(TClusterMember::ES_LEADER));
    ASSERT_THAT(second->OrderId, Lock_.OrderId());
    ASSERT_THAT(second->CurrentLeader, ToString(*Lock_.CurrentHolder()));

    Lock_.ReleaseFor(secondIdx);
    Lock_.AcquireFor(Idx_);
    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));

    ASSERT_THAT(ClusterMember_->State, Eq(TClusterMember::ES_LEADER));
    ASSERT_THAT(ClusterMember_->OrderId, Lock_.OrderId());
    ASSERT_THAT(ClusterMember_->CurrentLeader, ToString(*Lock_.CurrentHolder()));
    ASSERT_THAT(second->State, Eq(TClusterMember::ES_FOLLOWER));
    ASSERT_THAT(second->OrderId, Lock_.OrderId());
    ASSERT_THAT(second->CurrentLeader, ToString(*Lock_.CurrentHolder()));
}

TEST_F(TLeaderElectionTest, TriesToRecoverAfterError) {
    ASSERT_THAT(LockProxy_->AcquireCalls, 1);
    Lock_.ErrorFor(Idx_);

    bool hasWakeup{false};
    ActorSystem_->SetScheduledEventFilter([&hasWakeup] (auto& runtime, auto& ev, auto delay, auto&& deadline) {
        Y_UNUSED(runtime);
        Y_UNUSED(delay);
        Y_UNUSED(deadline);

        if (ev->GetTypeRewrite() == TEvents::TEvWakeup::EventType) {
            hasWakeup = true;
            return false;
        }

        return false;
    });

    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ActorSystem_->AdvanceCurrentTime(TDuration::Seconds(5));
    ASSERT_TRUE(hasWakeup);
    ActorSystem_->DispatchEvents({}, TDuration::Seconds(0));
    ASSERT_THAT(LockProxy_->AcquireCalls, 2);
}


class TLeaderElectionYdbTest: public ::testing::Test {
public:
    void SetUp() override {
        Db_.Reset(new TYdb);
        Db_->Init();
        ActorSystem_.Reset(new TCoordinationActorRuntime);
        ActorSystem_->Initialize();
    }

    void TearDown() override {
    }

protected:
    THolder<TYdb> Db_;
    IDistributedLockPtr Lock_;
    THolder<TCoordinationActorRuntime> ActorSystem_;
};

TEST_F(TLeaderElectionYdbTest, SingleNodeAcquiresLock) {
    auto obj = MakeHolder<TClusterMember>();
    auto ptr = obj.Get();
    auto actor = CreateClusterMemberActor(
        Db_->MakeLock(0),
        std::move(obj)
    );

    ActorSystem_->Register(
        actor
    );

    ptr->WaitLeader();
}

TEST_F(TLeaderElectionYdbTest, LeadershipChange) {
    TVector<TClusterMember*> members;

    TActorId leader;
    for (auto i = 0; i < 2; ++i) {
        auto obj = MakeHolder<TClusterMember>();
        members.push_back(obj.Get());
        auto actor = CreateClusterMemberActor(
                Db_->MakeLock(0),
                std::move(obj)
        );

        auto leaderId = ActorSystem_->Register(
                actor
        );

        if (i == 0) {
            members.back()->WaitLeader();
            leader = leaderId;
        } else {
            members[i]->WaitFollower();
        }
    }

    ASSERT_THAT(members[0]->State, TClusterMember::ES_LEADER);
    ASSERT_THAT(members[1]->State, TClusterMember::ES_FOLLOWER);

    ActorSystem_->Send(new IEventHandle{leader, {}, new TEvents::TEvPoisonPill});
    members[1]->WaitLeader();
}

TEST_F(TLeaderElectionYdbTest, CanRecoverAfterLockFailure) {
    Db_.Reset(new TYdb);
    auto lock = Db_->MakeLock(0);

    auto obj = MakeHolder<TClusterMember>();
    auto ptr = obj.Get();
    auto actor = CreateClusterMemberActor(
            Db_->MakeLock(0),
            std::move(obj),
            TDuration::Seconds(1)
    );

    const auto aid = ActorSystem_->Register(
            actor
    );
    ActorSystem_->EnableScheduleForActor(aid);
    ActorSystem_->WaitForBootstrap();

    Sleep(TDuration::Seconds(5));
    Db_->Init();

    ptr->WaitLeader();
}
