#include <solomon/libs/cpp/coordination/load_balancer/load_balancer.h>
#include <solomon/libs/cpp/actors/interconnect/init.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>

#include <library/cpp/actors/core/actorsystem.h>
#include <library/cpp/actors/core/executor_pool_basic.h>
#include <library/cpp/actors/core/log.h>
#include <library/cpp/actors/core/scheduler_basic.h>

#include <library/cpp/actors/interconnect/poller_tcp.h>
#include <library/cpp/actors/util/should_continue.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/system/hostname.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NCoordination;
using namespace NMonitoring;


struct TListenerActor final: public TActor<TListenerActor> {
    TListenerActor()
        : TActor<TListenerActor>{&TThis::StateFunc}
    {
    }

    STFUNC(StateFunc) {
        Y_UNUSED(ctx);
        switch (ev->GetTypeRewrite()) {
            hFunc(TLoadBalancerEvents::TEvAssignment, OnAssignment);
        }
    }

    void OnAssignment(const TLoadBalancerEvents::TEvAssignment::TPtr& ev) {
        TStringBuilder sb;
        sb << "received assignments: \n";
        sb << ev->Get()->Assignment;

        Cerr << sb;
    }
};

static TProgramShouldContinue ShouldContinue;

void OnTerminate(int) {
    ShouldContinue.ShouldStop();
}

THolder<TActorSystemSetup> BuildActorSystemSetup(ui32 threads)
{
    Y_VERIFY(threads > 0 && threads < 100);

    auto setup = MakeHolder<TActorSystemSetup>();

    setup->ExecutorsCount = 1;
    setup->Executors.Reset(new TAutoPtr<IExecutorPool>[1]);
    setup->Executors[0] = new TBasicExecutorPool(0, threads, 50);
    setup->Scheduler = new TBasicSchedulerThread(TSchedulerConfig(512, 0));

    return setup;
}

int main(int argc, char** argv) {
#ifdef _unix_
    signal(SIGPIPE, SIG_IGN);
#endif
    signal(SIGINT, &OnTerminate);
    signal(SIGTERM, &OnTerminate);

    TMetricRegistry registry;

    using namespace NLastGetopt;
    using namespace NYdb;

    TString ydbEndpoint;
    TString ydbDatabase;
    TString nodePath;
    TString semaphoreName;
    ui32 nodeIdx{0};

    auto opts = TOpts::Default();
    opts.AddLongOption('e', "endpoint", "YDB endpoint")
        .RequiredArgument("ENDPOINT")
        .StoreResult(&ydbEndpoint)
        .Required();

    opts.AddLongOption('d', "database", "YDB database")
        .RequiredArgument("DATABASE")
        .StoreResult(&ydbDatabase);

    opts.AddLongOption('n', "node", "Coordination node path")
        .RequiredArgument("PATH")
        .StoreResult(&nodePath)
        .Required();

    opts.AddLongOption('s', "semaphore", "Semaphore name")
        .RequiredArgument("NAME")
        .StoreResult(&semaphoreName)
        .Required();

    opts.AddLongOption('i', "index", "This node index")
        .RequiredArgument("INDEX")
        .StoreResult(&nodeIdx)
        .Required();

    NLastGetopt::TOptsParseResult result{&opts, argc, argv};

    TDriverConfig ydbConf;
    ydbConf.SetEndpoint(ydbEndpoint)
        .SetDatabase(ydbDatabase);

    TDriver driver{ydbConf};

    ui16 basePort{14324};
    TVector<TIcEndpoint> icNodes{
        {FQDNHostName(), "::1", basePort++, 1},
        {FQDNHostName(), "::1", basePort++, 2},
        {FQDNHostName(), "::1", basePort++, 3},
    };

    auto setup = BuildActorSystemSetup(1);

    TInterconnectConfig interconnectConfig{
        .ThisNodeId = nodeIdx,
        .Endpoints = std::move(icNodes)
    };

    InitInterconnect(*setup, interconnectConfig, std::shared_ptr<TMetricRegistry>(&registry));

    TYdbLockConfig lockConfig{
        .Driver = driver,
        .Path = nodePath,
        .Name = semaphoreName,
        .Data = FQDNHostName(),
    };

    TLoadBalancerConfig balancerConfig{
        .MetricFactory = registry,
        .PingInterval = TDuration::Seconds(5),
        .OfflineThreshold = TDuration::Seconds(30),
    };

    auto* balancer = CreateLoadBalancerActor(
        CreateStaticBalancer(),
        CreateYdbLock(lockConfig),
        std::move(balancerConfig)
    );

    setup->LocalServices.emplace_back(
        MakeLoadBalancerId(nodeIdx),
        TActorSetupCmd{balancer, TMailboxType::ReadAsFilled, 0}
    );

    auto actorSystem = MakeHolder<TActorSystem>(setup);
    actorSystem->Start();

    while (ShouldContinue.PollState() == TProgramShouldContinue::Continue) {
        Sleep(TDuration::MilliSeconds(200));
    }

    actorSystem->Stop();
    actorSystem->Cleanup();

    return ShouldContinue.GetReturnCode();
}
