#include "cluster_state.h"

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NCoordination::TClusterState& clusterState) {
    auto nodes = clusterState.Nodes();
    auto leader = clusterState.Leader();

    os << "Cluster leader is " << leader.NodeId << '(' << leader.Hostname << ") " << leader.OrderId << '\n';

    for (auto&& node: nodes) {
        os << node.NodeId() << '(' << node.Hostname() << ") " << node.State() << '\n';
    }

    return os;
}

namespace NSolomon::NCoordination {
    TNodeState::TNodeState(TNodeId nodeId)
        : NodeId_{nodeId}
    {
    }

    TNodeState::TNodeState(TNodeId nodeId, TString hostname)
        : NodeId_{nodeId}
        , Hostname_{std::move(hostname)}
    {
    }

    TNodeId TNodeState::NodeId() const {
        return NodeId_;
    }

    const TString& TNodeState::Hostname() const {
        return Hostname_;
    }

    ENodeState TNodeState::State() const {
        return State_;
    }

    void TNodeState::SetState(ENodeState state) {
        State_ = state;
    }

    bool TNodeState::IsOk() const {
        return State() == ENodeState::Ok;
    }

    bool operator==(const TNodeState& lhs, const TNodeState& rhs) {
        return lhs.NodeId() == rhs.NodeId();
    }

    void TClusterState::AddNode(const TNodeState& node, const NActors::TEvInterconnect::TNodeInfo& nodeInfo, TInstant s) {
        auto [it, isNew] = Nodes_.emplace(
            std::piecewise_construct,
            std::forward_as_tuple(node.NodeId()),
            std::forward_as_tuple(node, std::move(nodeInfo), s)
        );

        Y_UNUSED(it);
        Y_VERIFY_DEBUG(isNew, "Node is already present in the cluster");
    }

    void TClusterState::UpdateNode(TNodeId nodeId, ENodeState nodeState, TInstant ts) {
        if (auto* node = Nodes_.FindPtr(nodeId)) {
            node->State.SetState(nodeState);
            node->Timestamp = ts;
        } else {
            Y_VERIFY_DEBUG(false, "Trying to update node which is not yet present in the cluster %ull", nodeId);
        }
    }

    void TClusterState::UpdateNode(TNodeState nodeState, TInstant ts) {
        if (auto* node = Nodes_.FindPtr(nodeState.NodeId())) {
            node->State = std::move(nodeState);
            node->Timestamp = ts;
        } else {
            Y_VERIFY_DEBUG(false, "Trying to update node which is not yet present in the cluster %ull", nodeState.NodeId());
        }
    }

    void TClusterState::MarkUnavailable(TNodeId nodeId, TInstant ts) {
        UpdateNode(nodeId, ENodeState::NotAvailable, ts);
    }

    TVector<TNodeState> TClusterState::AliveNodes() const {
        TVector<TNodeState> nodes;
        nodes.reserve(Nodes_.size());

        for (auto&& [id, node]: Nodes_) {
            if (node.State.IsOk()) {
                nodes.push_back(node.State);
            }
        }

        return nodes;
    }

    TVector<TNodeState> TClusterState::Nodes() const {
        TVector<TNodeState> nodes;
        nodes.reserve(Nodes_.size());

        for (auto&& [id, node]: Nodes_) {
            nodes.push_back(node.State);
        }

        return nodes;
    }

    const TNodeState* TClusterState::FindById(TNodeId id) const {
        auto it = Nodes_.find(id);
        return (it == Nodes_.end()) ? nullptr : &it->second.State;
    }

    const TNodeState* TClusterState::FindByHostname(const TString& hostname) const {
        auto it = FindIf(Nodes_.begin(), Nodes_.end(), [&] (auto&& node) {
            return node.second.State.Hostname() == hostname;
        });

        return it == Nodes_.end()
            ? nullptr
            : &it->second.State;
    }

    const TLeaderInfo& TClusterState::Leader() const {
        return Leader_;
    }

    void TClusterState::SetLeader(const TString& hostname, ui64 orderId) {
        Leader_.Hostname = hostname;
        Leader_.OrderId = orderId;
        if (auto* node = FindByHostname(hostname)) {
            Leader_.NodeId = node->NodeId();
        }
    }
} // namespace NSolomon::NCoordination
