#pragma once

#include "node_state.h"

#include <solomon/libs/cpp/actors/fwd.h>

#include <library/cpp/actors/core/interconnect.h>

#include <util/datetime/base.h>
#include <util/generic/algorithm.h>
#include <util/generic/hash_set.h>
#include <util/generic/vector.h>
#include <util/string/builder.h>

namespace NSolomon::NCoordination {
    using TNodeId = ui32;
    class TNodeState {
    public:
        explicit TNodeState(TNodeId nodeId);
        TNodeState(TNodeId nodeId, TString hostname);

        TNodeId NodeId() const;
        const TString& Hostname() const;
        ENodeState State() const;
        void SetState(ENodeState state);
        bool IsOk() const;

    private:
        ENodeState State_{ENodeState::Unknown};
        ui32 NodeId_{0};
        TString Hostname_;
    };

    bool operator==(const TNodeState& lhs, const TNodeState& rhs);

    struct TLeaderInfo {
        TLeaderInfo() = default;
        TLeaderInfo(TNodeId nodeId, TString hostname, ui64 orderId)
            : Hostname{std::move(hostname)}
            , NodeId{nodeId}
            , OrderId{orderId}
        {
        }

        bool IsValid() const {
            return Hostname && NodeId != 0;
        }

        bool IsNewerThan(ui64 orderId) const {
            return OrderId > orderId;
        }

        TString Hostname;
        TNodeId NodeId{0};
        ui64 OrderId{0};
    };

    class TClusterState {
    public:
        TClusterState() = default;

        void AddNode(const TNodeState& node, const NActors::TEvInterconnect::TNodeInfo& nodeInfo, TInstant s);
        void UpdateNode(TNodeId nodeId, ENodeState nodeState, TInstant ts);
        void UpdateNode(TNodeState nodeState, TInstant ts);
        void MarkUnavailable(TNodeId nodeId, TInstant ts);

        TVector<TNodeState> AliveNodes() const;
        TVector<TNodeState> Nodes() const;

        const TNodeState* FindById(TNodeId id) const;
        const TNodeState* FindByHostname(const TString& hostname) const;

        const TLeaderInfo& Leader() const;
        void SetLeader(const TString& hostname, ui64 orderId);

        template <typename TFunc>
        void ForEachNode(TFunc&& fn) {
            for (auto&& [id, state]: Nodes_) {
                Y_UNUSED(id);
                fn(state.State, state.Timestamp);
            }
        }

    private:
        struct TNodeStateEx {
            TNodeStateEx(TNodeState s, const NActors::TEvInterconnect::TNodeInfo& node, TInstant ts)
                : State{std::move(s)}
                , Timestamp{ts}
                , NodeInfo{std::move(node)}
            {
            }

            TNodeState State;
            TInstant Timestamp;
            NActors::TEvInterconnect::TNodeInfo NodeInfo;
        };

        THashMap<TNodeId, TNodeStateEx> Nodes_;
        TLeaderInfo Leader_;
    };
} // namespace NSolomon::NCoordination

IOutputStream& operator<<(IOutputStream& os, const NSolomon::NCoordination::TClusterState& clusterState);
