#include "load_balancer.h"
#include "private_events.h"
#include "counters.h"

#include <solomon/libs/cpp/coordination/leader_election/leader.h>
#include <solomon/libs/cpp/logging/logging.h>

#include <library/cpp/actors/interconnect/interconnect.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/log.h>

using namespace NActors;

namespace NSolomon::NCoordination {
namespace {
    enum class EState {
        Initial = 0,
        Follower,
        Leader,
        Suspended,
    };

    class TLoadBalancerActor
        : public TActorBootstrapped<TLoadBalancerActor>
        , public IClusterMember
    {
        struct TProxy: IClusterMember {
            TProxy(IClusterMember* impl)
                : Impl_{impl}
            {
            }

            void OnBecomeLeader() override {
                Impl_->OnBecomeLeader();
            }

            void OnBecomeFollower() override {
                Impl_->OnBecomeFollower();
            }

            void OnLeaderChanged(TString leaderId, ui64 orderId) override {
                Impl_->OnLeaderChanged(std::move(leaderId), orderId);
            }

            void OnError(const TString& message) override {
                Impl_->OnError(message);
            }

        private:
            IClusterMember* Impl_{nullptr};
        };

    public:
        TLoadBalancerActor(ILoadBalancingStrategyPtr loadBalancer, IDistributedLockPtr lock, TLoadBalancerConfig config)
            : Counters_{config.MetricFactory}
            , Lock_{std::move(lock)}
            , LoadBalancer_{std::move(loadBalancer)}
            , Interval_{config.PingInterval}
            , OfflineThreshold_{config.OfflineThreshold}
        {
        }

        void Bootstrap() {
            Send(GetNameserviceActorId(), new TEvInterconnect::TEvListNodes);
            Become(&TThis::StateInit);
        }

        void OnNodesInfo(const TEvInterconnect::TEvNodesInfo::TPtr& ev) {
            auto&& nodes = ev->Get()->Nodes;
            const auto now = TActivationContext::Now();

            TClusterState cluster;
            MON_INFO(LoadBalancer, "load balancer nodes info:");
            for (auto& node: nodes) {
                MON_INFO(LoadBalancer, "host:" << node.ResolveHost << ", id:" << node.NodeId);
                TNodeState state{node.NodeId, node.ResolveHost};
                state.SetState(ENodeState::Ok);
                cluster.AddNode(state, std::move(node), now);
            }

            Cluster_ = std::move(cluster);

            auto proxy = MakeHolder<TProxy>(this);
            ClusterMemberId_ = RegisterWithSameMailbox(CreateClusterMemberActor(
                Lock_,
                std::move(proxy)
            ));

            BecomeFollower();
            PingSelf();
            Counters_.Resume();
            Counters_.SetTotalNodes(nodes.size());
        }

        STFUNC(StateInit) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvInterconnect::TEvNodesInfo, OnNodesInfo);
                hFunc(TEvents::TEvSubscribe, OnSubscribe);
                hFunc(TEvents::TEvUnsubscribe, OnUnsubscribe);
                hFunc(TLoadBalancerEvents::TEvDescribeRequest, OnDescribe);
                cFunc(TEvents::TEvPoisonPill::EventType, PassAway);
            }
        }

        STFUNC(StateFollower) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                sFunc(TEvents::TEvWakeup, OnWakeup);
                hFunc(TLoadBalancerPrivate::TEvAssignment, OnAssignments);
                hFunc(TEvInterconnect::TEvNodeDisconnected, OnNodeDisconnected);
                hFunc(TEvents::TEvUndelivered, OnUndelivered);

                hFunc(TLoadBalancerEvents::TEvDescribeRequest, OnDescribe);
                hFunc(TEvents::TEvSubscribe, OnSubscribe);
                hFunc(TEvents::TEvUnsubscribe, OnUnsubscribe);

                hFunc(TEvents::TEvPing, OnPing);
                hFunc(TLoadBalancerPrivate::TEvNodeInfo, OnNodeInfo);

                cFunc(TLoadBalancerEvents::TEvSuspend::EventType, OnSuspend);

                cFunc(TEvents::TEvPoisonPill::EventType, PassAway);
            }
        }

        STFUNC(StateLeader) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                sFunc(TEvents::TEvWakeup, OnWakeup);
                hFunc(TLoadBalancerPrivate::TEvAssignment, OnAssignments);
                hFunc(TEvInterconnect::TEvNodeDisconnected, OnNodeDisconnected);

                hFunc(TLoadBalancerEvents::TEvDescribeRequest, OnDescribe);
                hFunc(TEvents::TEvSubscribe, OnSubscribe);
                hFunc(TEvents::TEvUnsubscribe, OnUnsubscribe);

                hFunc(TEvents::TEvPing, OnPing);
                hFunc(TLoadBalancerPrivate::TEvNodeInfo, OnNodeInfo);

                cFunc(TLoadBalancerEvents::TEvSuspend::EventType, OnSuspend);

                cFunc(TEvents::TEvPoisonPill::EventType, PassAway);
            }
        }

        STFUNC(StateSuspended) {
            Y_UNUSED(ctx);
            switch (ev->GetTypeRewrite()) {
                cFunc(TLoadBalancerEvents::TEvResume::EventType, OnResume);
                cFunc(TEvents::TEvPoisonPill::EventType, PassAway);
                hFunc(TEvents::TEvSubscribe, OnSubscribe);
                hFunc(TEvents::TEvUnsubscribe, OnUnsubscribe);
                hFunc(TLoadBalancerEvents::TEvDescribeRequest, OnDescribe);
            }
        }

    private:
        void OnPing(const TEvents::TEvPing::TPtr& ev) {
            Send(
                ev->Sender,
                new TLoadBalancerPrivate::TEvNodeInfo,
                IEventHandle::FlagTrackDelivery | IEventHandle::FlagSubscribeOnSession
            );
        }

        void OnSubscribe(const TEvents::TEvSubscribe::TPtr& ev) {
            Listeners_.emplace(ev->Sender);
        }

        void OnUnsubscribe(const TEvents::TEvUnsubscribe::TPtr& ev) {
            Listeners_.erase(ev->Sender);
        }

        void OnSuspend() {
            MON_INFO(LoadBalancer, "Is now suspended");
            BecomeSuspended();
            Send(ClusterMemberId_, new TEvents::TEvPoisonPill);
            ClusterMemberId_ = {};
        }

        void OnResume() {
            MON_INFO(LoadBalancer, "Resuming");
            Bootstrap();
        }

        void OnNodeInfo(const TLoadBalancerPrivate::TEvNodeInfo::TPtr& ev) {
            const auto nodeId = ev->Sender.NodeId();
            Cluster_.UpdateNode(nodeId, ENodeState::Ok, TActivationContext::Now());
        }

        void OnDescribe(const TLoadBalancerEvents::TEvDescribeRequest::TPtr& ev) {
            auto sender = ev->Sender;

            Send(sender, new TLoadBalancerEvents::TEvDescribeResponse{
                State_,
                Cluster_,
                CurrentAssignments_,
            });
        }

        void OnBecomeLeader() override {
            if (IsSuspended()) {
                return;
            }

            MON_WARN(LoadBalancer, "Is now leader");
            BecomeLeader();
        }

        void OnBecomeFollower() override {
            if (IsSuspended()) {
                return;
            }

            MON_WARN(LoadBalancer, "Is now follower");
            BecomeFollower();
        }

        void OnError(const TString& message) override {
            MON_ERROR(LoadBalancer, "Lock lost with an error: " << message);

            if (IsLeader()) {
                BecomeFollower();
            }
        }

        void OnWakeup() {
            PingNodes();
            UpdateNodeStates();
            UpdateSelfState();

            if (IsLeader()) {
                MakeAssignments();
            }

            PingSelf();
        }

        void OnLeaderChanged(TString leaderId, ui64 orderId) override {
            const bool isNewer = !Cluster_.Leader().IsValid()
                || orderId > Cluster_.Leader().OrderId;

            Y_VERIFY_DEBUG(isNewer);
            if (!isNewer) {
                MON_WARN(LoadBalancer, "Leader changed to " << leaderId << ", but it's order id "
                    << orderId << " is <= than the current " << Cluster_.Leader().OrderId);
            }

            MON_INFO(LoadBalancer, "Leader changed to " << leaderId << " with order id " << orderId);
            Counters_.SetLeaderOrderId(orderId);
            Cluster_.SetLeader(leaderId, orderId);
        }

        void ApplyAssignments(NProto::TEvAssignment&& record) {
            auto&& leader = Cluster_.Leader();

            if (leader.OrderId != record.GetOrderId()) {
                MON_WARN(LoadBalancer, "Received assignments from node " << record.GetNodeId()
                    << " with order id " << record.GetOrderId()
                    << " which does not match current leader info "
                    << leader.Hostname << " order id " << leader.OrderId);

                return;
            }

            MON_INFO(LoadBalancer, "Received assignments from " << record.GetNodeId() << " with order id " << record.GetOrderId());

            auto assignments = AssignmentsFromProto(record);
            UpdateAssignmentTimestamp();
            for (auto listener: Listeners_) {
                Send(listener, new TLoadBalancerEvents::TEvAssignment{
                    assignments
                });
            }

            CurrentAssignments_ = std::move(assignments);
        }

        void OnAssignments(const TLoadBalancerPrivate::TEvAssignment::TPtr& ev) {
            if (!IsLeader() || SelfId().NodeId() == ev->Sender.NodeId()) {
                ApplyAssignments(std::move(ev->Get()->Record));
            }
        }

        void OnNodeDisconnected(const TEvInterconnect::TEvNodeDisconnected::TPtr& ev) {
            const auto nodeId = ev->Get()->NodeId;
            MON_INFO(LoadBalancer, "Disconnected node " << nodeId);
            Cluster_.MarkUnavailable(nodeId, TActivationContext::Now());
        }

        void OnUndelivered(TEvents::TEvUndelivered::TPtr ev) {
            const auto nodeId = ev->Sender.NodeId();
            MON_INFO(LoadBalancer, "Undelivered to node " << nodeId
                    << " (sender: " << ev->Sender.NodeId() << ", recipient: " << ev->Recipient.NodeId() << ")");
            Cluster_.MarkUnavailable(nodeId, TActivationContext::Now());
        }

        void PingSelf() {
            Schedule(Interval_, new TEvents::TEvWakeup);
        }

        void MakeAssignments() {
            auto nodes = Cluster_.AliveNodes();
            MON_INFO(LoadBalancer, "see " << nodes.size() << " alive nodes");

            auto assignments = LoadBalancer_->MakeAssignments(nodes);

            for (auto& assignment: assignments) {
                auto nodeId = assignment.NodeId();

                auto* nodeState = Cluster_.FindById(nodeId);
                Y_VERIFY(nodeState, "cannot find node %d", nodeId);

                MON_INFO(LoadBalancer, "assign " << assignment.Slices().size()
                        << " slices to node " << nodeState->Hostname() << " (" << nodeId << ") "
                        << nodeState->State());

                Send(
                    MakeLoadBalancerId(nodeId),
                    AssignmentsToProto(assignment, nodeId, Cluster_.Leader().OrderId),
                    IEventHandle::FlagTrackDelivery | IEventHandle::FlagSubscribeOnSession
                );
            }
        }

        void PingNodes() {
            auto&& nodes = Cluster_.Nodes();
            for (auto&& node: nodes) {
                auto balancerId = MakeLoadBalancerId(node.NodeId());
                Send(
                    balancerId,
                    new TEvents::TEvPing,
                    IEventHandle::FlagTrackDelivery | IEventHandle::FlagSubscribeOnSession
                );
            }
        }

        void UpdateNodeStates() {
            const auto now = TActivationContext::Now();
            ui32 onlineNodes{0};

            Cluster_.ForEachNode([&] (auto&& state, auto ts) {
                if (state.IsOk() && ((now - ts) > OfflineThreshold_)) {
                    Cluster_.MarkUnavailable(state.NodeId(), now);
                }

                if (state.IsOk()) {
                    ++onlineNodes;
                }
            });

            Counters_.SetOnlineNodes(onlineNodes);
        }

        void UpdateSelfState() {
            if (!AssignmentTimestamp_) {
                return;
            }

            const TInstant now = TActivationContext::Now();
            const TDuration assignmentAge = now - AssignmentTimestamp_;
            Counters_.SetAssignmentAgeSeconds(assignmentAge.Seconds());

            // we haven't received updates from the leader a while
            // so we'll cancel any old assignments for now
            if (assignmentAge > OfflineThreshold_) {
                MON_WARN(LoadBalancer, "Last assignment ts is " << AssignmentTimestamp_
                    << ". Cancelling any assignments until new ones are received");

                for (auto listener: Listeners_) {
                    Send(listener, new TLoadBalancerEvents::TEvAssignment{
                        TAssignments::EmptyAssignments(SelfId().NodeId()),
                    });
                }
            }
        }

        void PassAway() override {
            MON_WARN(LoadBalancer, "Dying");
            const auto thisNode = SelfId().NodeId();
            auto* as = TActorContext::ActorSystem();
            Cluster_.ForEachNode([&] (auto&& state, auto) {
                auto nodeId = state.NodeId();
                if (nodeId != thisNode) {
                    const auto interconnectProxy = as->InterconnectProxy(nodeId);
                    Send(interconnectProxy, new TEvents::TEvUnsubscribe());
                }
            });

            Send(ClusterMemberId_, new TEvents::TEvPoisonPill);
            IActor::PassAway();
        }

        void BecomeSuspended() {
            Become(&TThis::StateSuspended);
            State_ = ELoadBalancerState::Suspended;
            Counters_.Suspend();
        }

        void BecomeFollower() {
            Become(&TThis::StateFollower);
            State_ = ELoadBalancerState::Follower;
            Counters_.UnsetLeader();
        }

        void BecomeLeader() {
            Become(&TThis::StateLeader);
            State_ = ELoadBalancerState::Leader;
            Counters_.SetLeader();
        }

        bool IsSuspended() const {
            return State_ == ELoadBalancerState::Suspended;
        }

        bool IsLeader() const {
            return State_ == ELoadBalancerState::Leader;
        }

        ui32 ThisNode() const {
            return SelfId().NodeId();
        }

        void UpdateAssignmentTimestamp() {
            const auto now = TActivationContext::Now();
            AssignmentTimestamp_ = now;
            Counters_.SetAssignmentAgeSeconds(0);
        }

    private:
        TBalancerCounters Counters_;
        IDistributedLockPtr Lock_;
        THashSet<TActorId> Listeners_;
        ILoadBalancingStrategyPtr LoadBalancer_;
        TClusterState Cluster_;
        TDuration Interval_{TDuration::Seconds(5)};
        TDuration OfflineThreshold_{TDuration::Seconds(30)};
        TInstant AssignmentTimestamp_;
        TActorId ClusterMemberId_;
        ELoadBalancerState State_{ELoadBalancerState::Initial};
        TAssignments CurrentAssignments_{TAssignments::EmptyAssignments(0)};
    };

} // namespace
    TActorId MakeLoadBalancerId(ui32 nodeId) {
        static const char NAME[12] = "ldblncr";
        return TActorId{nodeId, NAME};
    }

    NActors::IActor* CreateLoadBalancerActor(
        ILoadBalancingStrategyPtr balancingStrategy,
        IDistributedLockPtr lock,
        TLoadBalancerConfig config)
    {
        return new TLoadBalancerActor{
            std::move(balancingStrategy),
            std::move(lock),
            std::move(config),
        };
    }
} // namespace NSolomon::NCoordination
