#include "load_balancer.h"

namespace NSolomon::NCoordination {
namespace {
    constexpr ui64 KEY_SPACE = Max<ui64>();
    constexpr ui64 KEY_SPACE_START = Min<ui64>();

    class TStaticBalancing final: public ILoadBalancingStrategy {
    public:
        TStaticBalancing(ui32 shardCount)
            : ShardCount_{shardCount}
        {
            Y_ENSURE(ShardCount_ != 0, "Shard count cannot be 0");
            Y_ENSURE((ShardCount_ & (ShardCount_ - 1)) == 0, "Shard count must be a power of 2");

            Slices_ = MakeSlices();
        }

        TVector<TAssignments> MakeAssignments(TVector<TNodeState> nodes) noexcept override {
            if (nodes.empty()) {
                return {};
            } else if (nodes.size() == 1) {
                return {
                    {nodes[0].NodeId(), {std::make_pair(KEY_SPACE_START, KEY_SPACE)}},
                };
            }

            Sort(nodes.begin(), nodes.end(), [] (auto&& lhs, auto&& rhs) {
                return lhs.NodeId() < rhs.NodeId();
            });

            const auto nodeCount = nodes.size();
            TVector<TSlices> nodeSlices(nodeCount);

            for (auto i = 0u; i < Slices_.size(); ++i) {
                auto nodeIdx = i % nodeCount;
                nodeSlices[nodeIdx].push_back(Slices_[i]);
            }

            TVector<TAssignments> result;

            for (auto i = 0u; i < nodeCount; ++i) {
                result.emplace_back(
                    nodes[i].NodeId(),
                    std::move(nodeSlices[i])
                );
            }

            return result;
        }

    private:
        TSlices MakeSlices() {
            ui64 sliceSize = KEY_SPACE / ShardCount_;
            TSlices result;

            ui64 sliceStart{0};
            ui64 sliceEnd{0};

            for (ui32 i = 0; i < ShardCount_; ++i) {
                sliceEnd = sliceStart + sliceSize;
                result.emplace_back(sliceStart, sliceEnd);
                sliceStart = sliceEnd + 1;
            }

            return result;
        }

    private:
        const ui32 ShardCount_;
        TSlices Slices_;
    };

} // namespace
    ILoadBalancingStrategyPtr CreateStaticBalancer(ui32 shardCount) {
        return MakeHolder<TStaticBalancing>(shardCount);
    }
} // namespace NSolomon::TCoordination
