#include <solomon/libs/cpp/coordination/load_balancer/load_balancer.h>
#include <solomon/libs/cpp/coordination/load_balancer/private_events.h>

#include <solomon/libs/cpp/coordination/testlib/actor_runtime.h>
#include <solomon/libs/cpp/coordination/testlib/lock.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NMonitoring;
using namespace NSolomon::NCoordination;
using namespace NSolomon::NTesting;
using namespace NThreading;

namespace NSolomon::NCoordination {
    std::ostream& operator<<(std::ostream& os, ELoadBalancerState state) {
        os << ToString(state);
        return os;
    }

    std::ostream& operator<<(std::ostream& os, ENodeState state) {
        os << ToString(state);
        return os;
    }

    std::ostream& operator<<(std::ostream& os, const TAssignments& assn) {
        os << ToString(assn);
        return os;
    }
} // namespace NSolomon::NCoordination

class TLoadBalancerTest: public testing::Test {
    void SetUp() override {
        Registry_.Reset(new TMetricRegistry);
        Runtime_.Reset(new TCoordinationActorRuntime);

        Lock_.Reset(new TMockLock);

        TLoadBalancerConfig config{
            .MetricFactory = *Registry_,
            .PingInterval = TDuration::MilliSeconds(50),
            .OfflineThreshold = TDuration::MilliSeconds(300),
        };

        for (auto idx = 0u; idx < Runtime_->GetNodeCount(); ++idx) {
            auto* balancer = CreateLoadBalancerActor(
                CreateStaticBalancer(),
                Lock_->CreateLockClient(idx),
                config
            );

            const auto nodeId = NodeId(idx);
            auto serviceId = MakeLoadBalancerId(nodeId);
            Runtime_->AddLocalService(
                serviceId,
                TActorSetupCmd{balancer, TMailboxType::Simple, 0},
                idx
            );

            Balancers_.push_back(serviceId);
        }

        Runtime_->Initialize();


        for (auto i = 0u; i < Runtime_->GetNodeCount(); ++i) {
            auto listenerId = Runtime_->AllocateEdgeActor(i);
            Runtime_->Send(new IEventHandle{
                Balancers_[i],
                listenerId, new TEvents::TEvSubscribe
            }, i);

            Listeners_.push_back(listenerId);
        }

        EnableSchedule();
        Sleep(TDuration::MilliSeconds(100));
    }

    void TearDown() override {
        Balancers_.clear();
        Listeners_.clear();
        Runtime_.Reset();
        Lock_.Reset();
        Registry_.Reset();
    }

protected:
    THolder<TLoadBalancerEvents::TEvDescribeResponse> Describe(ui32 nodeIdx) {
        auto listener = Listeners_[nodeIdx];
        auto nodeId = Runtime_->GetNodeId(nodeIdx);
        Runtime_->Send(new IEventHandle{
            MakeLoadBalancerId(nodeId), listener,
            new TLoadBalancerEvents::TEvDescribeRequest,
        }, nodeIdx);

        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerEvents::TEvDescribeResponse>(listener);
        return ev->Release();
    }

    void Suspend(ui32 nodeIdx) {
        auto nodeId = NodeId(nodeIdx);
        Runtime_->Send(new IEventHandle{
            MakeLoadBalancerId(nodeId), {}, new TLoadBalancerEvents::TEvSuspend
        }, nodeIdx);
    }

    void Resume(ui32 nodeIdx) {
        auto nodeId = NodeId(nodeIdx);
        Runtime_->Send(new IEventHandle{
            MakeLoadBalancerId(nodeId), {}, new TLoadBalancerEvents::TEvResume
        }, nodeIdx);
    }

    THolder<TLoadBalancerPrivate::TEvNodeInfo> PingAndWait(ui32 nodeIdx) {
        auto nodeId = NodeId(nodeIdx);
        auto listener = Listeners_[nodeIdx];

        Runtime_->Send(MakeLoadBalancerId(nodeId), listener, MakeHolder<TEvents::TEvPing>());
        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerPrivate::TEvNodeInfo>(listener);
        return ev->Release();
    }

    THolder<TLoadBalancerPrivate::TEvAssignment> MakeAssignments(ui32 nodeId, ui64 orderId, const TSlices& slices) {
        auto assn = MakeHolder<TLoadBalancerPrivate::TEvAssignment>();
        assn->Record.SetNodeId(nodeId);
        assn->Record.SetOrderId(orderId);
        for (auto&& s: slices) {
            auto* slice = assn->Record.AddSlices();
            slice->SetBegin(s.first);
            slice->SetEnd(s.second);
        }

        return assn;
    }

    // blocks assignments generated by actual balancers, but allows sending them from edge actor
    void BlockAssignments(TActorId edge) {
        Runtime_->SetEventFilter([&] (auto&, auto& event) {
            if (event->GetTypeRewrite() == TLoadBalancerPrivate::EvAssignment) {
                return event->Sender != edge;
            }

            return false;
        });
    }

    void EnableSchedule() {
        for (auto i = 0u; i < Balancers_.size(); ++i) {
            const auto actorId = Runtime_->GetLocalServiceId(Balancers_[i], i);
            Runtime_->EnableScheduleForActor(
                actorId,
                true
            );
        }
    }

    ui32 NodeId(ui32 nodeIdx) const {
        return Runtime_->GetNodeId(nodeIdx);
    }

protected:
    THolder<TMetricRegistry> Registry_;
    THolder<TMockLock> Lock_;
    THolder<TCoordinationActorRuntime> Runtime_;
    TVector<TActorId> Listeners_;
    TVector<TActorId> Balancers_;
};

TEST_F(TLoadBalancerTest, BalancerSeesOtherNodes) {
    Sleep(TDuration::MilliSeconds(500));
    Lock_->AcquireFor(0);
    for (auto i = 0u; i < Balancers_.size(); ++i) {
        auto ev = Describe(i);
        auto&& nodes = ev->ClusterState.Nodes();
        ASSERT_EQ(nodes.size(), 3u);
    }
}

TEST_F(TLoadBalancerTest, LeaderIsElected) {
    Lock_->AcquireFor(0);
    {
        auto ev = Describe(0);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Leader);
    }

    {
        auto ev = Describe(1);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Follower);
    }
}

TEST_F(TLoadBalancerTest, BalancerCanBeSuspended) {
    Lock_->AcquireFor(0);
    {
        auto ev = Describe(0);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Leader);
    }

    Suspend(0);
    {
        auto ev = Describe(0);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Suspended);
    }
}

TEST_F(TLoadBalancerTest, BalancerCanBeResumed) {
    Lock_->AcquireFor(0);
    {
        auto ev = Describe(0);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Leader);
    }

    Suspend(0);
    Lock_->AcquireFor(1);
    {
        auto ev = Describe(0);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Suspended);
    }
    {
        auto ev = Describe(1);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Leader);
    }

    Resume(0);
    {
        Sleep(TDuration::MilliSeconds(50));
        auto ev = Describe(0);
        ASSERT_EQ(ev->BalancerState, ELoadBalancerState::Follower);
    }
}

TEST_F(TLoadBalancerTest, BalancerRespondsToPings) {
    for (auto i = 0u; i < Runtime_->GetNodeCount(); ++i) {
        auto listener = Listeners_[i];
        const auto nodeId = Runtime_->GetNodeId(i);

        Runtime_->Send(new IEventHandle{MakeLoadBalancerId(nodeId), listener, new TEvents::TEvPing,}, i);
        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerPrivate::TEvNodeInfo>(listener);
        Y_UNUSED(ev);
    }
}

TEST_F(TLoadBalancerTest, SuspendedNodeIsDetected) {
    Lock_->AcquireFor(0);
    Suspend(1);

    Sleep(TDuration::MilliSeconds(500));

    auto ev = Describe(0);

    auto&& nodes = ev->ClusterState.Nodes();
    auto nodeId = Runtime_->GetNodeId(1);
    auto it = FindIf(nodes.begin(), nodes.end(), [nodeId] (auto&& node) {
        return node.NodeId() == nodeId;
    });

    ASSERT_TRUE(it != nodes.end());
    ASSERT_EQ(it->State(), ENodeState::NotAvailable);
}

TEST_F(TLoadBalancerTest, ListenerReceivesAssignments) {
    auto leaderId = Balancers_[0];
    Lock_->AcquireFor(0);
    BlockAssignments(leaderId);

    for (auto i = 0u; i < Balancers_.size(); ++i) {
        auto assn = MakeAssignments(0, Lock_->OrderId(), {{3, 42}});
        const auto expected = assn->Record;

        Runtime_->Send(new IEventHandle{Balancers_[i], leaderId, assn.Release()}, i);

        const auto listenerId = Listeners_[i];
        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerEvents::TEvAssignment>(listenerId);
        ASSERT_EQ(ev->Get()->Assignment, AssignmentsFromProto(expected));
    }
}

TEST_F(TLoadBalancerTest, LeaderAcceptsOnlyOwnAssignments) {
    auto leaderIdx = 0;
    auto leaderId = Balancers_[leaderIdx];
    Lock_->AcquireFor(leaderIdx);
    BlockAssignments(leaderId);

    auto assn1 = MakeAssignments(0, Lock_->OrderId(), {{3, 42}});
    auto assn2 = MakeAssignments(0, Lock_->OrderId(), {{3, 42}});
    const auto expected = assn1->Record;

    {
        auto nodeId = leaderIdx + 1; // not local in the leader's perspective
        Runtime_->Send(new IEventHandle{leaderId, Balancers_[1], assn1.Release()}, nodeId);
        const auto listenerId = Listeners_[leaderIdx];
        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerEvents::TEvAssignment>(listenerId);
        ASSERT_NE(ev->Get()->Assignment, AssignmentsFromProto(expected));
    }

    {
        Runtime_->Send(new IEventHandle{leaderId, leaderId, assn2.Release()}, leaderIdx);
        const auto listenerId = Listeners_[leaderIdx];
        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerEvents::TEvAssignment>(listenerId, TDuration::Seconds(10));
        ASSERT_EQ(ev->Get()->Assignment, AssignmentsFromProto(expected));
    }
}

TEST_F(TLoadBalancerTest, NoAssignmentsWhileSuspended) {
    Suspend(0);
    Runtime_->Send(MakeLoadBalancerId(0), {}, std::move(MakeAssignments(0, 0, {{4, 52}})));

    TEventsList evs = Runtime_->CaptureEvents();
    ASSERT_FALSE(FindEvent<TLoadBalancerEvents::TEvAssignment>(evs));
}

TEST_F(TLoadBalancerTest, BalancerRespondsToPingsFromOtherNode) {
    auto edge = Runtime_->AllocateEdgeActor(1);
    for (auto idx = 0u; idx < Runtime_->GetNodeCount(); ++idx) {
        const auto nodeId = Runtime_->GetNodeId(idx);
        Runtime_->Send(new IEventHandle{MakeLoadBalancerId(nodeId), edge, new TEvents::TEvPing}, 1);
        auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerPrivate::TEvNodeInfo>(edge);
        Y_UNUSED(ev);
    }
}

TEST_F(TLoadBalancerTest, DeadNodeIsDetected) {
    Lock_->AcquireFor(0);
    auto nodeId = Runtime_->GetNodeId(2);
    Runtime_->Send(MakeLoadBalancerId(nodeId), {}, MakeHolder<TEvents::TEvPoison>());

    Sleep(TDuration::MilliSeconds(500));

    auto description = Describe(0);
    auto&& clusterState = description->ClusterState;
    ASSERT_EQ(clusterState.AliveNodes().size(), 2u);

    auto&& nodes = clusterState.Nodes();

    auto it = FindIf(nodes.begin(), nodes.end(), [&] (auto&& node) {
        return node.NodeId() == nodeId;
    });

    ASSERT_NE(it, nodes.end());
    EXPECT_FALSE(it->IsOk());
    EXPECT_EQ(it->State(), ENodeState::NotAvailable);
}

TEST_F(TLoadBalancerTest, StaleAssignmentsAreIgnored) {
    auto edge = Runtime_->AllocateEdgeActor(2);
    BlockAssignments(edge);

    const auto leaderIdx = 1;
    Lock_->AcquireFor(leaderIdx);
    Runtime_->Send(Balancers_[2], edge, std::move(MakeAssignments(0, 0, TSlices{{2, 3}})));

    const auto orderId = Lock_->OrderId();
    auto assn = MakeAssignments(leaderIdx, orderId, {{3, 42}});
    const auto expected = assn->Record;
    Runtime_->Send(Balancers_[2], edge, std::move(assn));

    auto ev = Runtime_->GrabEdgeEvent<TLoadBalancerEvents::TEvAssignment>(Listeners_[2]);
    ASSERT_EQ(ev->Get()->Assignment, AssignmentsFromProto(expected));
}
