#include <solomon/libs/cpp/distributed_lock/lock.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>
#include <ydb/public/sdk/cpp/client/ydb_scheme/scheme.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/sighandler/async_signals_handler.h>

#include <array>
#include <atomic>

using namespace NYdb;
using namespace NYdb::NCoordination;
using namespace NYdb::NScheme;
using namespace NSolomon;
using namespace NLastGetopt;

static std::atomic<bool> ShouldStop = false;

class TCommandBase {
public:
    void Execute(int argc, char** argv) {
        NLastGetopt::TOpts opts = Opts();
        NLastGetopt::TOptsParseResult result{&opts, argc, argv};

        TDriverConfig conf;
        conf.SetEndpoint(YdbEndpoint_)
            .SetDatabase(YdbDatabase_);

        Driver_ = {conf};
        Client_ = {Driver_};

        DoExecute();
    }

    virtual ~TCommandBase() = default;
    virtual NLastGetopt::TOpts Opts() {
        NLastGetopt::TOpts opts;
        opts.AddLongOption('e', "endpoint", "YDB endpoint")
            .RequiredArgument("ENDPOINT")
            .StoreResult(&YdbEndpoint_)
            .Required();

        opts.AddLongOption('d', "database", "YDB database")
            .RequiredArgument("DATABASE")
            .StoreResult(&YdbDatabase_);

        opts.AddLongOption('n', "node", "Coordination node path")
            .RequiredArgument("PATH")
            .StoreResult(&NodePath_)
            .Required();

        opts.AddLongOption('s', "semaphore", "Semaphore name")
            .RequiredArgument("NAME")
            .StoreResult(&SemaphoreName_)
            .Required();


        return opts;
    }

    virtual TStringBuf Name() const = 0;
    virtual TStringBuf Help() const = 0;

private:
    virtual void DoExecute() = 0;

public:
    const TString& SemaphoreName() const {
        return SemaphoreName_;
    }

    const TString& NodePath() const {
        return NodePath_;
    }

    TClient& Client() {
        return *Client_;
    }

    TDriver& Driver() {
        return *Driver_;
    }

protected:
    TString YdbEndpoint_;
    TString YdbDatabase_;
    TString NodePath_;
    TString SemaphoreName_;
    std::optional<TDriver> Driver_;
    std::optional<TClient> Client_;
};

template <typename TDerived>
struct TSemaphoreOps {
    void GetSession(TSessionSettings settings = {}) {
        auto result = Self().Client()
            .StartSession(Self().NodePath(), std::move(settings))
            .ExtractValueSync();

        if (!result.IsSuccess()) {
            ythrow yexception() << result.GetIssues().ToString();
        }

        Session_ = result.GetResult();
    }

    std::optional<TSemaphoreDescription> DescribeSemaphore(TDescribeSemaphoreSettings settings = {}) {
        auto result = Session()
            .DescribeSemaphore(Self().SemaphoreName(), std::move(settings))
            .ExtractValueSync();

        if (result.IsSuccess()) {
            return result.GetResult();
        } else if (result.GetStatus() == EStatus::NOT_FOUND) {
            return std::nullopt;
        } else {
            ythrow yexception() << result.GetStatus() << ": " << result.GetIssues().ToString();
        }
    }

    void CreateSemaphore() {
        auto result = Session()
            .CreateSemaphore(Self().SemaphoreName(), 1)
            .ExtractValueSync();

        if (!result.IsSuccess()) {
            ythrow yexception() << result.GetIssues().ToString();
        }
    }

    TSession Session() const {
        return Session_;
    }

private:
    TDerived& Self() {
        return static_cast<TDerived&>(*this);
    }

private:
    TSession Session_;
};

class TCreateCommand: public TCommandBase, public TSemaphoreOps<TCreateCommand> {
public:
    void DoExecute() override {
        auto& client = Client();


        if (Recursive_) {
            EnsureDirectoryExists(NodePath());
        }

        {
            // create if does not exist
            Cout << "Creating node " << NodePath_ << Endl;
            auto result = client.CreateNode(NodePath(), {}).ExtractValueSync();
            if (!result.IsSuccess()) {
                ythrow yexception() << result.GetIssues().ToString();
            }

            Cout << "Done." << Endl;
        }

        {
            GetSession();
            auto&& result = DescribeSemaphore();
            if (result) {
                Cout << "Semaphore " << SemaphoreName() << " already exists" << Endl;
                return;
            }

            Cout << "Creating semaphore " << SemaphoreName() << Endl;
            CreateSemaphore();
            Cout << "Done." << Endl;
        }
    }

    void EnsureDirectoryExists(TStringBuf nodePath) {
        const auto directory = TString{nodePath.RBefore('/')};

        TSchemeClient client{Driver()};
        {
            auto result = client.DescribePath(directory).ExtractValueSync();
            if (result.IsSuccess()) {
                return;
            }

            Y_ENSURE(result.GetStatus() != NYdb::EStatus::NOT_FOUND, "Unexpected error: " << result.GetIssues().ToString());
        }

        {
            auto result = client.MakeDirectory(directory, {}).ExtractValueSync();
            Y_ENSURE(result.IsSuccess(), "Cannot create directory " << directory << ": " << result.GetIssues().ToString());
        }
    }

    TStringBuf Name() const override {
        return TStringBuf("create");
    }

    TStringBuf Help() const override {
        return TStringBuf("Create a coordination node if required and a semaphore on that node");
    }

    TOpts Opts() override {
        auto opts = TCommandBase::Opts();
        opts.AddLongOption('p', "parents", "Create parent directories if they don't exist")
            .StoreTrue(&Recursive_);
        return opts;
    }

private:
    bool Recursive_{false};
};

class TWatchCommand: public TCommandBase, public TSemaphoreOps<TWatchCommand> {
    struct TSemaphoreWatcher: TThrRefBase {
        TSemaphoreWatcher(TSession session)
            : Session_{std::move(session)}
        {
        }

        void Watch(const TString& semaphore, std::atomic<bool>& stop) {
            bool fired{false};
            Cout << "Watching semaphore " << semaphore << "\n";

            while (!stop) {
                if (fired) {
                    auto description = Describe(semaphore);
                    PrintTo(description, Cout);
                }

                fired = ChangedEv_.WaitT(TDuration::MilliSeconds(500));
            }
        }

        void PrintTo(const TSemaphoreDescription& description, IOutputStream& os) {
            auto&& owners = description.GetOwners();
            auto&& waiters = description.GetWaiters();

            auto hasOwner = !owners.empty();

            static const TString EMPTY{"<empty>"};

            os << "OrderId: " << (hasOwner ? ToString(owners[0].GetOrderId()) : EMPTY) << '\n'
                << "Owner: " << (hasOwner ? ToString(owners[0].GetSessionId()) : EMPTY) << '\n'
                << "Owner Data: " << (hasOwner ? owners[0].GetData() : EMPTY) << '\n'
                << "Data: " << description.GetData() << '\n'
                << "Waiters count: " << waiters.size() << '\n'
                << "------\n";
        }

    private:
        void OnChanged(bool) {
            ChangedEv_.Signal();
        }

        TSemaphoreDescription Describe(const TString& semaphore) {
            auto settings = TDescribeSemaphoreSettings{}
                .OnChanged([self=TIntrusivePtr{this}] (bool changed) { self->OnChanged(changed); })
                .WatchData()
                .WatchOwners()
                .IncludeOwners()
                .IncludeWaiters();

            auto result = Session_.DescribeSemaphore(semaphore, std::move(settings)).ExtractValueSync();

            if (!result.IsSuccess()) {
                ythrow yexception() << result.GetIssues().ToString();
            }

            return result.GetResult();
        }

    private:
        TSession Session_;
        TAutoEvent ChangedEv_;
    };

public:
    static void OnChanged(ESessionState state) {
        Cout << "Session state changed to " << state << Endl;
    };

    static void OnStopped() {
        Cout << "Session stopped. Exiting..." << Endl;
    }

    void DoExecute() override {
        Cout << "Watching semaphore " << SemaphoreName() << " on node " << NodePath() << Endl;

        GetSession(TSessionSettings{}
            .OnStateChanged(&TWatchCommand::OnChanged)
            .OnStopped(&TWatchCommand::OnStopped)
        );

        auto watcher = MakeIntrusive<TSemaphoreWatcher>(Session());
        try {
            watcher->Watch(SemaphoreName(), ShouldStop);
        } catch (...) {
            Cerr << "An error while trying to describe semaphore: " << CurrentExceptionMessage();
            throw;
        }
    }

    TStringBuf Name() const override {
        return TStringBuf("watch");
    }

    TStringBuf Help() const override {
        return TStringBuf("Watch the given semaphore");
    }
};


using TCommands = std::array<THolder<TCommandBase>, 2>;

void PrintUsage(TStringBuf executable, TCommands& commands) {
    Cerr << executable << " <command> [-e|--endpoint] [-d|--database] [-n|--node] [-s|--semaphore] [OPTIONS]\n\n";

    for (auto& command: commands) {
        Cerr << command->Name() << '\t' << command->Help() << '\n';
    }
}

bool IsHelp(TStringBuf arg) {
    return arg == "-h" || arg == "--help" || arg == "help";
}

void Stop(int) {
    ShouldStop = true;
}

int main(int argc, char** argv) {
    SetAsyncSignalHandler(SIGTERM, &Stop);
    SetAsyncSignalHandler(SIGINT, &Stop);

    TCommands commands{{
        MakeHolder<TCreateCommand>(),
        MakeHolder<TWatchCommand>(),
    }};

    if (argc <= 1 || IsHelp(argv[1])) {
        PrintUsage(argv[0], commands);
        return 0;
    }

    auto it = FindIf(commands.begin(), commands.end(), [arg = TStringBuf{argv[1]}] (auto&& cmd){
        return cmd->Name() == arg;
    });

    if (it == commands.end()) {
        Cerr << "Unknown command: " << argv[1];
        PrintUsage(argv[0], commands);
        return 1;
    }

    try {
        (*it)->Execute(argc, argv);
    } catch (...) {
        Cerr << CurrentExceptionMessage();
        return 1;
    }

    return 0;
}
