#include <solomon/libs/cpp/distributed_lock/lock.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>

#include <util/generic/guid.h>
#include <util/stream/file.h>
#include <util/string/cast.h>
#include <util/string/strip.h>

using namespace NSolomon;
using namespace NYdb;
using namespace NYdb::NCoordination;

struct TYdb {
public:
    TYdb()
        : Database{Strip(TFileInput("ydb_database.txt").ReadAll())}
        , Driver{
            TDriverConfig{}
                .SetEndpoint(Strip(TFileInput("ydb_endpoint.txt").ReadAll()))
                .SetDatabase(Database)
        }
        , Client{Driver}
    {
    }

    TString Database;
    TDriver Driver;
    TClient Client;
};

enum EState {
    ES_UNKNOWN = 0,
    ES_LOCKED = 1,
    ES_UNLOCKED = 2,
    ES_DEFUNCT = 3,
};

struct TLockStateListener: public ILockStateListener  {
private:
    TLockStateListener() {
    }

public:
    static TIntrusivePtr<TLockStateListener> Create() {
        class TImpl: public TLockStateListener {};

        return MakeIntrusive<TImpl>();
    }

    void OnLock() override {
        State.store(ES_LOCKED);
        LockEv.Signal();
    }

    void OnUnlock() override {
        State.store(ES_UNLOCKED);
        UnlockEv.Signal();
    }

    void OnChanged(const TLockDescription& desc) override {
        CurrentDescription = desc;
        ChangeEv.Signal();
    }

    void OnError(TString message) override {
        ErrorMessage = std::move(message);
        State.store(ES_DEFUNCT);
        ErrorEv.Signal();
    }

    bool WaitLock(TDuration timeout = {}) {
        if (timeout) {
            return LockEv.WaitT(timeout);
        }

        LockEv.WaitI();
        return true;
    }

    bool WaitUnlock(TDuration timeout = {}) {
        if (timeout) {
            return UnlockEv.WaitT(timeout);
        }

        UnlockEv.WaitI();
        return true;
    }

    bool WaitChanged(TDuration timeout = {}) {
        if (timeout) {
            return ChangeEv.WaitT(timeout);
        }

        ChangeEv.WaitI();
        return true;
    }

    void WaitError() {
        ErrorEv.WaitI();
    }

    std::atomic<EState> State{ES_UNKNOWN};
    TLockDescription CurrentDescription;
    TAutoEvent LockEv;
    TAutoEvent UnlockEv;
    TAutoEvent ChangeEv;
    TAutoEvent ErrorEv;
    TString ErrorMessage;
};

struct TPrepareLockTest: public testing::Test {
    TYdb Db;
};

TEST_F(TPrepareLockTest, LongPath) {
    auto nodePath = "/" + Db.Database + "/x/y/z";
    auto lockName = CreateGuidAsString();

    PrepareYdbLock(TYdbLockConfig{
            .Driver = Db.Driver,
            .Path = nodePath,
            .Name = lockName,
            .Data = {},
    }).GetValueSync();
}

TEST_F(TPrepareLockTest, InvalidPath) {
    auto nodePath = "/nosuchdatabase/foo";
    auto lockName = CreateGuidAsString();

    ASSERT_THROW(PrepareYdbLock(TYdbLockConfig{
            .Driver = Db.Driver,
            .Path = nodePath,
            .Name = lockName,
            .Data = {},
    }).GetValueSync(), yexception);
}

class TYdbLockTest: public testing::Test {
public:
    void SetUp() override {
        NodePath_ = "/" + Db_.Database + "/foo";
        LockName_ = CreateGuidAsString();
        PrepareLock();

        Db_ = {};
    }

    void TearDown() override {
        StopYdbDriver();
    }

protected:
    IDistributedLockPtr MakeLock(TString data = {}) {
        return CreateYdbLock(TYdbLockConfig{
            .Driver = Db_.Driver,
            .Path = NodePath_,
            .Name = LockName_,
            .Data = std::move(data),
        });
    }

    void PrepareLock() {
        PrepareYdbLock(TYdbLockConfig{
            .Driver = Db_.Driver,
            .Path = NodePath_,
            .Name = LockName_,
            .Data = {},
        }).GetValueSync();
    }

    void SetNodePath(TString path) {
        NodePath_ = std::move(path);
    }

    void StopYdbDriver() {
        Db_.Driver.Stop(true);
    }

private:
    TString NodePath_;
    TString LockName_;
    TYdb Db_;
};

TEST_F(TYdbLockTest, SingleOwner) {
    auto l = TLockStateListener::Create();
    auto lock = MakeLock("foo");
    lock->Acquire(l);
    ASSERT_TRUE(l->WaitLock(TDuration::Seconds(20)));
    ASSERT_TRUE(l->WaitChanged(TDuration::Seconds(1)));
    ASSERT_EQ(l->State.load(), ES_LOCKED);
    ASSERT_EQ(l->CurrentDescription.Data, "foo");

    lock->Release();
    ASSERT_TRUE(l->WaitUnlock(TDuration::Seconds(20)));
    ASSERT_EQ(l->State.load(), ES_UNLOCKED);
}

// TODO: also make a test for a lock release on object destruction
TEST_F(TYdbLockTest, NewLockAfterRelease) {
    {
        auto l = TLockStateListener::Create();
        auto first = MakeLock();
        first->Acquire(l);
        ASSERT_TRUE(l->WaitLock(TDuration::Seconds(10)));
        ASSERT_EQ(l->State.load(), ES_LOCKED);

        first->Release();

        ASSERT_TRUE(l->WaitUnlock(TDuration::Seconds(20)));
        ASSERT_EQ(l->State.load(), ES_UNLOCKED);
    }

    {
        auto l = TLockStateListener::Create();
        auto second = MakeLock();
        second->Acquire(l);
        ASSERT_TRUE(l->WaitLock(TDuration::Seconds(20)));
        ASSERT_EQ(l->State.load(), ES_LOCKED);
    }
}

TEST_F(TYdbLockTest, SameLockAfterRelease) {
    auto firstListener = TLockStateListener::Create();
    auto secondListener = TLockStateListener::Create();

    auto lock = MakeLock();
    lock->Acquire(firstListener);
    ASSERT_TRUE(firstListener->WaitLock(TDuration::Seconds(20)));

    lock->Release();
    ASSERT_TRUE(firstListener->WaitUnlock(TDuration::Seconds(20)));

    lock->Acquire(secondListener);
    ASSERT_TRUE(secondListener->WaitLock(TDuration::Seconds(20)));
}

TEST_F(TYdbLockTest, DescribeLock) {
    {
        auto l = TLockStateListener::Create();
        auto lock = MakeLock("1");
        lock->Acquire(l);
        ASSERT_TRUE(l->WaitLock());
    }
    {
        auto l2 = TLockStateListener::Create();
        auto lock = MakeLock("2");
        lock->Acquire(l2);
        l2->WaitChanged();

        auto description = l2->CurrentDescription.Data;
        ASSERT_EQ(description, "1");
    }
}

TEST_F(TYdbLockTest, MultipleNotifications) {
    TVector<TIntrusivePtr<TLockStateListener>> listeners;
    TVector<IDistributedLockPtr> candidates;

    for (auto i = 0; i < 10; ++i) {
        candidates.push_back(MakeLock(ToString(i)));
        listeners.emplace_back(TLockStateListener::Create());
        auto listener = listeners.back();
        candidates.back()->Acquire(listener).Wait();
    }

    auto l = TLockStateListener::Create();
    auto lock = MakeLock();
    lock->Acquire(l).Wait();
    l->WaitChanged();
    auto prevOrderId = l->CurrentDescription.OrderId;

    for (auto i = 0u; i < candidates.size() - 1; ++i) {
        candidates[i]->Release();
    }

    ASSERT_TRUE(listeners.back()->WaitLock());
    do {
        l->WaitChanged();
        ASSERT_GT(l->CurrentDescription.OrderId, prevOrderId);
        prevOrderId = l->CurrentDescription.OrderId;
    } while (l->CurrentDescription.Data != ToString(candidates.size() - 1));
}

TEST_F(TYdbLockTest, InvalidClientSettings) {
    SetNodePath("/somethingbad");

    auto l = TLockStateListener::Create();
    auto lock = MakeLock();
    lock->Acquire(l);
    l->WaitError();
    ASSERT_EQ(l->State.load(), ES_DEFUNCT);
}

TEST_F(TYdbLockTest, DriverIsStopped) {
    auto l = TLockStateListener::Create();
    auto lock = MakeLock();
    lock->Acquire(l);

    StopYdbDriver();
    l->WaitError();
    ASSERT_EQ(l->State.load(), ES_DEFUNCT);
}

