#include "ydb_helpers.h"

#include <ydb/public/sdk/cpp/client/ydb_driver/driver.h>
#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>
#include <ydb/public/sdk/cpp/client/ydb_scheme/scheme.h>

#include <util/string/builder.h>
#include <util/string/split.h>

using namespace NYdb;
using namespace NYdb::NCoordination;
using namespace NYdb::NScheme;
using namespace NThreading;

namespace NSolomon {
    class TDirectoryCreationContext: public TAtomicRefCount<TDirectoryCreationContext> {
    public:
        static TDirectoryCreationContext* Create(TSchemeClient client, TString path) {
            return new TDirectoryCreationContext{std::move(client), std::move(path)};
        }

        TFuture<void> CreateRecursive() {
            AbsentDirPromise_.GetFuture().Subscribe([=] (auto f) {
                auto found = f.GetValue();
                if (!found) {
                    CompleteSuccess();
                    return;
                }

                CreateDirectory();
            });

            FindFirstAbsent();
            return Promise_;
        }

        ~TDirectoryCreationContext() = default;

    private:
        TDirectoryCreationContext(TSchemeClient client, TString path)
            : Path_{std::move(path)}
            , CurrentPos_{Path_.find('/', 1)}
            , Client_{std::move(client)}
        {
            Self_ = this;
        }

        void FindFirstAbsent() {
            Client_.DescribePath(CurrentSubdir()).Subscribe([=] (auto f) mutable {
                auto result = f.ExtractValue();
                const auto probablyExists = result.IsSuccess() || (result.GetStatus() == EStatus::UNAUTHORIZED);

                if (probablyExists && IsEnd()) {
                    AbsentDirPromise_.SetValue(false);
                } else if (probablyExists) {
                    Advance();
                    FindFirstAbsent();
                } else {
                    AbsentDirPromise_.SetValue(true);
                }
            });
        }

        bool IsEnd() const {
            return (CurrentPos_ == Path_.size() - 1) || CurrentPos_ == TString::npos;
        }

        void Advance() {
            CurrentPos_ = Path_.find('/', CurrentPos_ + 1);
        }

        TString CurrentSubdir() const {
            return Path_.substr(0, CurrentPos_);
        }

        void CreateDirectory() {
            if (IsEnd()) {
                IsLast_ = true;
            }

            Client_.MakeDirectory(CurrentSubdir(), {}).Apply([=] (auto f) mutable {
                auto result = f.ExtractValue();
                if (!result.IsSuccess()) {
                    CompleteError(result.GetIssues().ToString());
                } else if (IsLast_){
                    CompleteSuccess();
                } else {
                    Advance();
                    CreateDirectory();
                }
            });
        }

        void CompleteError(const TString& msg) {
            Promise_.SetException(msg);
            Dispose();
        }

        void CompleteSuccess() {
            Promise_.SetValue();
            Dispose();
        }

        void Dispose() {
            Self_.Reset();
        }

    private:
        TIntrusivePtr<TDirectoryCreationContext> Self_;
        TPromise<void> Promise_{NewPromise<void>()};
        TPromise<bool> AbsentDirPromise_{NewPromise<bool>()};
        const TString Path_;
        size_t CurrentPos_{0};
        TSchemeClient Client_;
        bool IsLast_{false};
    };

    TFuture<void> EnsureDirectoryExists(const TDriver& driver, const TString& path) {
        auto directory = TString{TStringBuf{path}.RBefore('/')};
        TSchemeClient client{driver};
        auto ctx = TDirectoryCreationContext::Create(std::move(client), std::move(directory));
        return ctx->CreateRecursive();
    }

    TFuture<void> CreateSemaphore(const TDriver& driver, const TString& nodePath, const TString& semaphoreName) {
        TClient client{driver};

        return client.CreateNode(nodePath, {})
            .Apply([client, nodePath] (auto f) mutable {
                auto result = f.ExtractValue();
                Y_ENSURE(result.IsSuccess(), result.GetIssues().ToString());
                return client.StartSession(nodePath, {});
            }).Apply([semaphoreName, client] (auto f) mutable {
                auto result = f.ExtractValue();
                Y_ENSURE(result.IsSuccess(), result.GetIssues().ToString());
                auto session = result.GetResult();
                return session.CreateSemaphore(semaphoreName, 1);
            }).Apply([client] (auto f) {
                Y_UNUSED(client);
                auto result = f.ExtractValue();
                Y_ENSURE(
                    result.IsSuccess() || result.GetStatus() == EStatus::ALREADY_EXISTS,
                    result.GetIssues().ToString()
                );
            });
    }
} // namespace NSolomon
