#include "lock.h"
#include "ydb_helpers.h"

#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>

#include <solomon/libs/cpp/error_or/error_or.h>

#include <util/system/hostname.h>
#include <util/system/spinlock.h>

using namespace NYdb::NCoordination;
using namespace NThreading;


namespace NSolomon {
namespace {
    const TString DEFAULT_DATA{FQDNHostName()};
    const TString DEFAULT_SEMAPHORE_NAME{"lock"};

    bool operator==(const TLockDescription& lhs, const TLockDescription& rhs) {
        return lhs.OrderId == rhs.OrderId && lhs.Data == rhs.Data;
    }

    class TYdbLock: public IDistributedLock, public std::enable_shared_from_this<TYdbLock> {
    private:
        enum class EState {
            Init,
            Locked,
            Unlocked,
            Error,
        };

        explicit TYdbLock(const TYdbLockConfig& config)
            : SessionProvider_{CreateSessionProvider(
                    TClient{config.Driver},
                    config.Path,
                    [wptr{weak_from_this()}] {
                        if (auto self = wptr.lock()) {
                            self->OnSessionStopped();
                        }
                    }
            )}
            , Data_{config.Data ? config.Data : DEFAULT_DATA}
            , SemaphoreName_{config.Name ? config.Name : DEFAULT_SEMAPHORE_NAME}
        {
        }

    public:
        static std::shared_ptr<TYdbLock> Create(const TYdbLockConfig& config) {
            struct TImpl: TYdbLock {
                explicit TImpl(const TYdbLockConfig& config)
                    : TYdbLock(config)
                {
                }
            };

            return std::make_shared<TImpl>(config);
        }

        ~TYdbLock() {
            // TODO: release a lock on destruction
        }

        void OnSessionStopped() const {
            auto g = Guard(Lock_);
            Y_VERIFY_DEBUG(StateListener_);

            State_ = EState::Error;
            if (StateListener_) {
                StateListener_->OnError(TString{"AlARM! session is closed"});
            }
        }

        TFuture<void> Acquire(ILockStateListenerPtr listener) override {
            auto g = Guard(Lock_);

            StateListener_ = listener;
            return SessionProvider_->GetSession().Apply([wptr{weak_from_this()}] (auto f) {
                auto self = wptr.lock();
                if (!self) {
                    return;
                }

                auto val = f.ExtractValue();

                if (val.Fail()) {
                    self->NotifyListenerError(val.Error().ToString());
                    return;
                }

                auto session = val.Extract();
                self->AcquireImpl(std::move(session));
            });
        }

        TFuture<void> Describe() {
            return SessionProvider_->GetSession().Apply([wptr{weak_from_this()}] (auto f) {
                auto self = wptr.lock();
                if (!self) {
                    return;
                }

                auto val = f.ExtractValue();

                if (val.Fail()) {
                    self->NotifyListenerError(val.Error().ToString());
                    return;
                }

                auto session = val.Extract();
                self->DescribeImpl(std::move(session));
            });
        }

        TFuture<void> Release() override {
            return SessionProvider_->GetSession().Apply([wptr{weak_from_this()}] (auto f) {
                auto self = wptr.lock();
                if (!self) {
                    return MakeFuture();
                }

                auto val = f.ExtractValue();

                if (val.Fail()) {
                    self->NotifyListenerError(val.Error().ToString());
                    return MakeFuture();
                }

                auto session = val.Extract();
                return self->ReleaseImpl(session);
            });
        }

        TFuture<void> ReleaseImpl(TSession session) {
            return session.ReleaseSemaphore(SemaphoreName_).Apply([wptr{weak_from_this()}] (auto) {
                auto self = wptr.lock();
                if (!self) {
                    return;
                }

                self->Unlock();
            });
        }

        void AcquireImpl(TSession session) {
            Y_ENSURE(session);
            session.AcquireSemaphore(SemaphoreName_, TAcquireSemaphoreSettings{}
                .Data(Data_)
                .Count(1)
                .OnAccepted([wptr{weak_from_this()}] {
                    auto self = wptr.lock();
                    if (!self) {
                        return;
                    }

                    self->Describe();
                })
            ).Apply([wptr{weak_from_this()}] (auto f) {
                auto self = wptr.lock();
                if (!self) {
                    return;
                }

                auto g = Guard(self->Lock_);
                auto result = f.GetValue();
                if (result.IsSuccess()) {
                    auto val = result.ExtractResult();
                    if (val) {
                        self->Lock();
                    }
                } else {
                    self->NotifyListenerError(result.GetIssues().ToString());
                }

                self->Describe();
            });
        }

    private:
        void Lock() const {
            Y_VERIFY_DEBUG(StateListener_);
            Y_VERIFY_DEBUG(State_ != EState::Locked);

            State_ = EState::Locked;

            if (StateListener_) {
                StateListener_->OnLock();
            }
        }

        void Unlock() const {
            Y_VERIFY_DEBUG(StateListener_);

            auto g = Guard(Lock_);

            if (State_ != EState::Locked) {
                return;
            }

            State_ = EState::Unlocked;

            if (StateListener_) {
                StateListener_->OnUnlock();
            }
        }

        void NotifyListenerOnChange(const TLockDescription& description) {
            Y_VERIFY_DEBUG(StateListener_);

            if (StateListener_) {
                StateListener_->OnChanged(description);
            }
        }

        void NotifyListenerError(TString message) {
            Y_VERIFY_DEBUG(StateListener_);

            if (State_ == EState::Locked) {
                StateListener_->OnUnlock();
            }

            State_ = EState::Error;

            if (StateListener_) {
                StateListener_->OnError(std::move(message));
            }
        }

        void OnChanged(bool) {
            Describe();
        }

        void DescribeImpl(TSession session) {
            session.DescribeSemaphore(SemaphoreName_, TDescribeSemaphoreSettings{}
                .WatchData()
                .WatchOwners()
                .IncludeOwners()
                .OnChanged([wptr{weak_from_this()}] (bool changed) {
                    if (auto self = wptr.lock()) {
                        self->OnChanged(changed);
                    }
                })
            )
            .Subscribe([wptr{weak_from_this()}] (auto f) {
                auto self = wptr.lock();
                if (!self) {
                    return;
                }

                auto result = f.ExtractValue();
                if (result.IsSuccess()) {
                    auto& owners = result.GetResult().GetOwners();
                    Y_VERIFY_DEBUG(owners.size() <= 1);
                    if (owners.size() < 1) {
                        // XXX error?
                        return;
                    }

                    auto& owner = owners[0];
                    TLockDescription result;
                    result.Data = owner.GetData();
                    result.OrderId = owner.GetOrderId();

                    {
                        auto g = Guard(self->Lock_);
                        // it was a spurious wakeup
                        if (result == self->CurrentDescription_) {
                            return;
                        }

                        self->CurrentDescription_ = result;
                        self->OwnerSession_ = owner;
                    }

                    self->NotifyListenerOnChange(std::move(result));
                } else {
                    self->NotifyListenerError(result.GetIssues().ToString());
                }
            });
        }

    private:
        TIntrusivePtr<ISessionProvider> SessionProvider_;

        mutable EState State_{EState::Init};

        const TString Data_;
        const TString Path_;
        const TString SemaphoreName_;
        ILockStateListenerPtr StateListener_{nullptr};

        TAdaptiveLock Lock_;
        TLockDescription CurrentDescription_;
        TSemaphoreSession OwnerSession_;
    };
} // namespace
    IDistributedLockPtr CreateYdbLock(const TYdbLockConfig& config) {
        return TYdbLock::Create(config);
    }

    NThreading::TFuture<void> PrepareYdbLock(const TYdbLockConfig& config) {
        using namespace NYdb;

        auto driver = config.Driver;

        return EnsureDirectoryExists(driver, config.Path)
            .Apply([driver, path{config.Path}, semaphoreName{config.Name}] (auto f) {
                try {
                    f.TryRethrow();
                } catch (...) {
                    return MakeErrorFuture<void>(std::current_exception());
                }

                return CreateSemaphore(driver, path, semaphoreName);
            });
    }
} // namespace NSolomon
