#include "ydb_helpers.h"

#include <ydb/public/sdk/cpp/client/ydb_coordination/coordination.h>

#include <solomon/libs/cpp/error_or/error_or.h>

#include <util/system/spinlock.h>
#include <util/system/condvar.h>

using namespace NYdb::NCoordination;
using namespace NThreading;

namespace NSolomon {
namespace {
    class TSessionProvider: public ISessionProvider {
    public:
        TSessionProvider(const TClient& client, TString path, std::function<void()> sessionCloseHandler)
            : Path_{std::move(path)}
            , SessionCloseHandler_{std::move(sessionCloseHandler)}
            , Client_{client}
        {
        }

        ~TSessionProvider() {
            Stop();
        }

        void Stop() override {
            auto g = Guard(Lock_);
            if (!CurrentSession_) {
                return;
            }

            IsClosing_ = true;
            // should trigger OnStopped
            CurrentSession_.Close();

            CloseEv_.WaitI(Lock_);
        }

        TFuture<TErrorOr<TSession, NYql::TIssues>> GetSession() override {
            auto g = Guard(Lock_);
            if (CurrentSession_) {
                return MakeFuture(TErrorOr<TSession, NYql::TIssues>::FromValue(CurrentSession_));
            } else if (CreateSessionPromise_.Initialized()) {
                return CreateSessionPromise_;
            }

            return CreateSession();
        }

    private:
        TFuture<TErrorOr<TSession, NYql::TIssues>> CreateSession() {
            CreateSessionPromise_ = NewPromise<TErrorOr<TSession, NYql::TIssues>>();

            Client_.StartSession(Path_, TSessionSettings{}
                .OnStopped([self{TIntrusivePtr{this}}] { self->OnStopped(); })
                .ConnectTimeout(CONNECTION_TIMEOUT)
            ).Subscribe([=] (auto f) mutable {
                auto val = f.ExtractValue();
                if (val.IsSuccess()) {
                    CurrentSession_ = val.GetResult();
                    CreateSessionPromise_.SetValue(TErrorOr<TSession, NYql::TIssues>::FromValue(CurrentSession_));
                } else {
                    CreateSessionPromise_.SetValue(TErrorOr<TSession, NYql::TIssues>::FromError(val.GetIssues()));
                }
            });

            return CreateSessionPromise_;
        }

        void OnSessionStarted(const TResult<TSession>& val) {
            auto g = Guard(Lock_);
            if (val.IsSuccess()) {
                CurrentSession_ = val.GetResult();
            }

            CreateSessionPromise_ = {};
        }

        void OnStopped() {
            auto g = Guard(Lock_);
            NotifySessionClose();
            CurrentSession_ = {};
            CreateSessionPromise_ = {};

            if (IsClosing_) {
                CloseEv_.Signal();
            }
        }

        void NotifySessionClose() {
            SessionCloseHandler_();
        }

    private:
        TMutex Lock_;
        TCondVar CloseEv_;

        const TString Path_;

        bool IsClosing_{false};
        TPromise<TErrorOr<TSession, NYql::TIssues>> CreateSessionPromise_;

        std::function<void()> SessionCloseHandler_;
        TClient Client_;
        TSession CurrentSession_;

        static constexpr TDuration CONNECTION_TIMEOUT = TDuration::Seconds(5);
    };
} // namespace

    TIntrusivePtr<ISessionProvider> CreateSessionProvider(
        const NYdb::NCoordination::TClient& client,
        TString path,
        std::function<void()> sessionCloseHandler)
    {
        return MakeIntrusive<TSessionProvider>(std::move(client), std::move(path), std::move(sessionCloseHandler));
    }
} // namespace NSolomon
