#pragma once

#include <util/generic/noncopyable.h>
#include <util/generic/yexception.h>

#include <variant>
#include <optional>

namespace NSolomon {
    struct TNotInitalizedException: public yexception {
    };

    struct TNoValueException: public yexception {
    };

    struct TGenericError {
    public:
        TGenericError(TString message)
            : Message_{std::move(message)}
        {
        }

        TStringBuf Message() const {
            return Message_;
        }

        TString MessageString() const {
            return Message_;
        }

    private:
        TString Message_;
    };

    template <typename TVal, typename TErr, bool Throw = false>
    class [[nodiscard]] TErrorOr: TMoveOnly {
        struct TValueTag{};
        struct TErrorTag{};

    public:
        template <typename... TArgs>
        static auto FromError(TArgs&&... args) {
            return TErrorOr<TVal, TErr, Throw>(TErrorTag{}, std::forward<TArgs>(args)...);
        }

        template <typename... TArgs>
        static auto FromValue(TArgs&&... args) {
            return TErrorOr<TVal, TErr, Throw>(TValueTag{}, std::forward<TArgs>(args)...);
        }

        TErrorOr() noexcept
            : Val_{std::in_place_index<0>, std::monostate{}}
        {
        }

        TErrorOr(const TVal& val) noexcept
            : Val_{std::in_place_index<1>, val}
        {
        }

        TErrorOr(TVal&& val) noexcept
            : Val_{std::in_place_index<1>, std::move(val)}
        {
        }

        TErrorOr(const TErr& err) noexcept
            : Val_{std::in_place_index<2>, err}
        {
        }

        TErrorOr(TErr&& err) noexcept
            : Val_{std::in_place_index<2>, std::move(err)}
        {
        }

        [[nodiscard]] bool Success() const noexcept (!Throw) {
            CheckInitialized();
            return Val_.index() == 1;
        }

        [[nodiscard]] bool Fail() const noexcept(!Throw) {
            return !Success();
        }

        [[nodiscard]] bool IsInitialized() const noexcept(!Throw) {
            return Val_.index() != 0;
        }

        [[nodiscard]] TVal& Value() noexcept (!Throw) {
            CheckValue();
            return std::get<1>(Val_);
        }

        [[nodiscard]] const TVal& Value() const noexcept (!Throw) {
            CheckValue();
            return std::get<1>(Val_);
        }

        [[nodiscard]] TVal Extract() noexcept (!Throw) {
            CheckValue();
            auto pval = std::get_if<1>(&Val_);

            TVal v = std::move(*pval);
            Val_.template emplace<0>(std::monostate{});

            return v;
        }

        [[nodiscard]] const TErr& Error() const noexcept (!Throw) {
            CheckError();
            return std::get<2>(Val_);
        }

        [[nodiscard]] TErr ExtractError() noexcept (!Throw) {
            CheckError();
            auto pval = std::get_if<2>(&Val_);
            TErr e = std::move(*pval);
            Val_.template emplace<0>(std::monostate{});

            return e;
        }

        template<typename T>
        TErrorOr<T, TErr> PassError() const {
            return TErrorOr<T, TErr>::FromError(std::get<2>(Val_));
        }

    private:
        template<typename... TArgs>
        TErrorOr(TValueTag, TArgs&&... args)
            : Val_{std::in_place_index<1>, std::forward<TArgs>(args)...}
        {
        }

        template<typename... TArgs>
        TErrorOr(TErrorTag, TArgs&&... args)
            : Val_{std::in_place_index<2>, std::forward<TArgs>(args)...}
        {
        }

        void CheckInitialized() const noexcept(!Throw) {
            if constexpr (Throw) {
                Y_ENSURE_EX(IsInitialized(), TNotInitalizedException());
            } else {
                Y_VERIFY(IsInitialized());
            }
        }

        void CheckValue() const noexcept (!Throw) {
            CheckInitialized();
            if constexpr (Throw) {
                Y_ENSURE_EX(Val_.index() == 1, TNoValueException());
            } else {
                Y_VERIFY(Val_.index() == 1);
            }
        }

        void CheckError() const noexcept (!Throw) {
            CheckInitialized();
            if constexpr (Throw) {
                Y_ENSURE_EX(Val_.index() == 2, TNoValueException());
            } else {
                Y_VERIFY(Val_.index() == 2);
            }
        }

    private:
        std::variant<std::monostate, TVal, TErr> Val_;
    };

    template <typename TErr, bool Throw>
    class [[nodiscard]] TErrorOr<void, TErr, Throw>: TMoveOnly {
        struct TErrorTag{};

    public:
        template <typename... TArgs>
        static auto FromError(TArgs&&... args) {
            return TErrorOr<void, TErr, Throw>(TErrorTag{}, std::forward<TArgs>(args)...);
        }

        static auto FromValue() {
            return TErrorOr<void, TErr, Throw>{};
        }

        TErrorOr() noexcept
            : Val_{std::in_place_index<1>, std::monostate{}}
        {
        }

        TErrorOr(const TErr& err) noexcept
            : Val_{std::in_place_index<2>, err}
        {
        }

        TErrorOr(TErr&& err) noexcept
            : Val_{std::in_place_index<2>, std::move(err)}
        {
        }

        [[nodiscard]] bool Success() const noexcept (!Throw) {
            CheckInitialized();
            return Val_.index() == 1;
        }

        [[nodiscard]] bool Fail() const noexcept(!Throw) {
            return !Success();
        }

        [[nodiscard]] bool IsInitialized() const noexcept(!Throw) {
            return Val_.index() != 0;
        }

        [[nodiscard]] const TErr& Error() const noexcept (!Throw) {
            CheckError();
            return std::get<2>(Val_);
        }

        [[nodiscard]] TErr ExtractError() noexcept (!Throw) {
            CheckError();
            auto pval = std::get_if<2>(&Val_);
            TErr e = std::move(*pval);
            Val_.template emplace<0>(std::monostate{});

            return e;
        }

        template<typename T>
        TErrorOr<T, TErr> PassError() {
            return TErrorOr<T, TErr>::FromError(std::get<2>(Val_));
        }

    private:
        template<typename... TArgs>
        TErrorOr(TErrorTag, TArgs&&... args)
            : Val_{std::in_place_index<2>, std::forward<TArgs>(args)...}
        {
        }

        void CheckInitialized() const noexcept(!Throw) {
            if constexpr (Throw) {
                Y_ENSURE_EX(IsInitialized(), TNotInitalizedException());
            } else {
                Y_VERIFY(IsInitialized());
            }
        }

        void CheckValue() const noexcept (!Throw) {
            CheckInitialized();
            if constexpr (Throw) {
                Y_ENSURE_EX(Val_.index() == 1, TNoValueException());
            } else {
                Y_VERIFY(Val_.index() == 1);
            }
        }

        void CheckError() const noexcept (!Throw) {
            CheckInitialized();
            if constexpr (Throw) {
                Y_ENSURE_EX(Val_.index() == 2, TNoValueException());
            } else {
                Y_VERIFY(Val_.index() == 2);
            }
        }

    private:
        std::variant<std::monostate, std::monostate, TErr> Val_;
    };
} // namespace NSolomon
