#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;

struct TValue {
    int Foo = 42;

    TValue()
        : Foo(42)
    {
    }

    TValue(int foo)
        : Foo(foo)
    {
    }
};

struct TErr {
    enum EType {
        Something,
        SomethingElse,
    };

    TErr(EType type, TString msg)
        : Type{type}
        , Message{std::move(msg)}
    {
    }

    EType Type;
    TString Message;
};

TEST(TErrorOrTest, Simple) {
    using TStatus = TErrorOr<TValue, TErr, true>;

    auto err = TStatus::FromError(TErr::SomethingElse, "SomethingElse happened!");
    ASSERT_TRUE(!err.Success());
    auto& e = err.Error();
    ASSERT_EQ(e.Type, TErr::SomethingElse);
    ASSERT_EQ(e.Message, "SomethingElse happened!");
    ASSERT_THROW((void) err.Value(), TNoValueException);

    auto val = TStatus::FromValue(43);

    ASSERT_THROW((void) val.Error(), yexception);
    ASSERT_TRUE(val.Success());
    ASSERT_EQ(val.Value().Foo, 43);
}

TEST(TErrorOrTest, Initalization) {
    {
        TErrorOr<TValue, TErr, true> e;
        ASSERT_THROW((void) e.Success(), TNotInitalizedException);
    }
    {
        TErrorOr<TValue, TErr, true> e;
        ASSERT_THROW((void) e.Value(), TNotInitalizedException);
    }
    {
        TErrorOr<TValue, TErr, true> e;
        ASSERT_THROW((void) e.Error(), TNotInitalizedException);
    }
}

TEST(TErrorOrTest, Construction) {
    using TStatus = TErrorOr<TValue, TErr, true>;
    auto val = TStatus::FromValue(TValue{1});
    ASSERT_EQ(val.Value().Foo, 1);

    auto val2 = TStatus{TValue{2}};
    ASSERT_EQ(val2.Value().Foo, 2);

    auto val3 = TStatus::FromValue(3);
    ASSERT_EQ(val3.Value().Foo, 3);

    TValue v{4};
    auto val4 = TStatus::FromValue(v);
    ASSERT_EQ(val4.Value().Foo, 4);

    v = TValue{4};
    auto val5 = TStatus{v};
    ASSERT_EQ(val5.Value().Foo, 4);
}

TEST(TErrorOrTest, MoveOnlyValue) {
    using TStatus = TErrorOr<THolder<int>, TErr, true>;

    TStatus s{MakeHolder<int>(42)};
    ASSERT_EQ(*s.Value(), 42);
    auto v = s.Extract();
    ASSERT_EQ(*v, 42);
    ASSERT_THROW((void) s.Value(), TNotInitalizedException);
}

TEST(TErrorOrTest, VoidStatus) {
    using TStatus = TErrorOr<void, TErr, true>;
    {
        auto val = TStatus::FromValue();
        ASSERT_TRUE(val.Success());
        ASSERT_THROW((void) val.Error(), TNoValueException);
    }
    {
        auto val = TStatus::FromError(TErr::Something, "foo");
        ASSERT_TRUE(!val.Success());
        ASSERT_EQ(val.Error().Message, "foo");
    }
}
