#pragma once

#include <library/cpp/monlib/metrics/ewma.h>
#include <library/cpp/monlib/metrics/metric.h>

namespace NSolomon {

// TODO(ivanzhukov@): SOLOMON-5622

constexpr auto EWMA_DEFAULT_INTERVAL = TDuration::Seconds(5);
const double EWMA_ALPHA_1MIN = 1. - std::exp(-static_cast<double>(EWMA_DEFAULT_INTERVAL.Seconds()) / 60. / 1.);

/**
 * EWMA -- https://en.wikipedia.org/wiki/Moving_average#Exponential_moving_average
 * Every implementation below first collects all data within some interval, and then computes a series element Y_i
 * in its own way
 */

/**
 * Every Y_i is computed as (sum / count) within an interval
 */
class TEWMAMean final: public NMonitoring::IExpMovingAverage {
public:
    TEWMAMean(NMonitoring::IGauge* metric, double alpha)
        : Metric_{metric}
        , Alpha_{alpha}
    {
        Y_VERIFY(metric != nullptr, "Passing nullptr metric is not allowed");
    }

    // This method is NOT thread-safe
    void Tick() override {
        const auto current = Uncounted_.fetch_and(0);
        auto count = Count_.fetch_and(0);

        const double instantRate = count == 0 ? 0 : (double(current) / count);

        if (Y_UNLIKELY(!IsInitialized())) {
            Metric_->Set(instantRate);
            Init_ = true;
        } else {
            const double currentRate = Metric_->Get();
            Metric_->Set(Alpha_ * (instantRate - currentRate) + currentRate);
        }
    }

    void Update(i64 value) override {
        Uncounted_ += value;
        ++Count_;
    }

    double Rate() const override {
        return Metric_->Get();
    }

    void Reset() override {
        Init_ = false;
        Uncounted_ = 0;
        Count_ = 0;
    }

private:
    bool IsInitialized() const {
        return Init_;
    }

private:
    std::atomic<i64> Uncounted_{0};
    std::atomic<ui64> Count_{0};
    std::atomic<bool> Init_{false};

    NMonitoring::IGauge* Metric_{nullptr};
    double Alpha_;
};

/**
 * Every Y_i is computed as sum within an interval
 */
class TEWMASum final: public NMonitoring::IExpMovingAverage {
public:
    TEWMASum(NMonitoring::IGauge* metric, double alpha)
        : Metric_{metric}
        , Alpha_{alpha}
    {
        Y_VERIFY(metric != nullptr, "Passing nullptr metric is not allowed");
    }

    // This method is NOT thread-safe
    void Tick() override {
        const auto currentY = Uncounted_.fetch_and(0);

        if (Y_UNLIKELY(!IsInitialized())) {
            Metric_->Set(currentY);
            Init_ = true;
        } else {
            const double prevY = Metric_->Get();
            Metric_->Set(Alpha_ * (currentY - prevY) + prevY);
        }
    }

    void Update(i64 value) override {
        Uncounted_ += value;
    }

    double Rate() const override {
        return Metric_->Get();
    }

    void Reset() override {
        Init_ = false;
        Uncounted_ = 0;
    }

private:
    bool IsInitialized() const {
        return Init_;
    }

private:
    std::atomic<i64> Uncounted_{0};
    std::atomic<bool> Init_{false};

    NMonitoring::IGauge* Metric_{nullptr};
    double Alpha_;
};

} // namespace NSolomon
