#pragma once

#include <util/datetime/base.h>

#include <cmath>

namespace NSolomon {

/**
 * Compute the EWMA (Exponential Weighted Moving Average) of a non-evenly-spaced time-series.
 *
 * The traditional formula for evenly-spaced time-series is:
 * x_n = w * x_n-1 + (1 - w) * x
 * x_n: estimation after n datapoint
 * w: learning rate (constant)
 * x: new datapoint
 *
 * But when the time-series is unevenly-spaced, the previous algorithm compute
 * a value which may represent a time window far greater that the expected one.
 * This algorithm used a slight modification of the previous one, it uses a
 * variable value of w, based on the elapsed time between the last two
 * datapoints.
 * We still have `x_n = w * x_n-1 + (1 - w) * x`
 * but w = exp(-dt/tau)
 * dt: time between the last two datapoints
 * tau: constant indicating the "window" of the average (longer window means
 * the average will need longer to converge)
 *
 * A very good source of theoritical knowledge about algorithm that works with
 * uneven time series can be found here:
 * http://www.eckner.com/papers/Algorithms%20for%20Unevenly%20Spaced%20Time%20Series.pdf
 */
class TEwma {
public:
    TEwma(TDuration window, TInstant now, double initialValue = 0.0) noexcept
        : Tau_{window.MicroSeconds() / M_LN2}
        , Timestamp_{now}
        , Value_{initialValue}
    {
    }

    void Add(TInstant now, double x) noexcept {
        const double delta = static_cast<double>((now - Timestamp_).MicroSeconds());
        const double w = std::exp(-delta / Tau_);
        Timestamp_ = now;
        Value_ = Value_ * w + (1 - w) * x;
    }

    void Reset(TInstant now, double value) noexcept {
        Timestamp_ = now;
        Value_ = value;
    }

    double WindowUs() const noexcept {
        return Tau_;
    }

    TInstant Timestamp() const noexcept {
        return Timestamp_;
    }

    double Value() const noexcept {
        return Value_;
    }

private:
    const double Tau_;
    TInstant Timestamp_;
    double Value_;
};

/**
 * The same EWMA (Exponential Weighted Moving Average) implementation, but uses atomic operations.
 */
class TAtomicEwma {
public:
    TAtomicEwma(TDuration window, TInstant now, double initialValue = 0.0) noexcept
        : Tau_{window.MicroSeconds() / M_LN2}
    {
        Timestamp_.store(now, std::memory_order_relaxed);
        Value_.store(initialValue, std::memory_order_relaxed);
    }

    void Add(TInstant now, double x) noexcept {
        TInstant prevTimestamp = Timestamp_.exchange(now);
        const double delta = static_cast<double>((now - prevTimestamp).MicroSeconds());
        const double w = std::exp(-delta / Tau_);
        Value_.store(Value_.load() * w + (1 - w) * x);
    }

    void Reset(TInstant now, double value) noexcept {
        Timestamp_.store(now, std::memory_order_relaxed);
        Value_.store(value, std::memory_order_relaxed);
    }

    double WindowUs() const noexcept {
        return Tau_;
    }

    TInstant Timestamp() const noexcept {
        return Timestamp_.load(std::memory_order_relaxed);
    }

    double Value() const noexcept {
        return Value_.load(std::memory_order_relaxed);
    }

private:
    const double Tau_;
    std::atomic<TInstant> Timestamp_;
    std::atomic<double> Value_;
};

} // namespace NSolomon
