#pragma once

#include <util/datetime/base.h>

#include <cmath>

namespace NSolomon {

/**
 * Compute a rate (in tick per second) based on an EWMA of the inter-arrival
 * time of the ticks. E.g. if the inter-arrival time is 100ms, it means the rate is 10 tick/s.
 */
template <typename TEwma>
class TEwmaRate {
public:
    TEwmaRate(TDuration window, TInstant now, double initialValue) noexcept
        : Ewma_{window, now, initialValue}
    {
    }

    TEwmaRate(TDuration window, TInstant now) noexcept
        : TEwmaRate(window, now, TDuration::Seconds(1).MicroSeconds())
    {
    }

    /**
     * Indicate that a tick happened
     */
    void Tick(TInstant now) noexcept {
        const auto delta = now - Ewma_.Timestamp();
        Ewma_.Add(now, static_cast<double>(delta.MicroSeconds()));
    }

    /**
     * Return the rate of ticks in second as a double.
     * It uses the current absence of tick in [lastArrivalTime_, now] as
     * a parameter to compute more accurate value. (i.e. if we don't see a tick
     * for a long time, the reported rate will still decrease)
     */
    double Rate(TInstant now) const noexcept {
        double ewma = Ewma_.Value();
        if (ewma == 0.0) {
            // EWMA is not yet initialized
            return 0.0;
        }

        const double delta = static_cast<double>((now - Ewma_.Timestamp()).MicroSeconds());
        if (delta > ewma) { // if too much time has passed, we decay the estimation
            double tau = Ewma_.WindowUs();
            double w = std::exp(-delta / tau);
            ewma = ewma * w + (1 - w) * delta;
        }
        return TDuration::Seconds(1).MicroSeconds() / ewma; // tick/sec
    }

    /**
     * Resets state of EWMA rate.
     */
    void Reset(TInstant now, double value = TDuration::Seconds(1).MilliSeconds()) noexcept {
        Ewma_.Reset(now, value);
    }

private:
    TEwma Ewma_;
};

} // namespace NSolomon
