#include "glob.h"

namespace NSolomon {
namespace {

// see https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions#Escaping
constexpr TStringBuf RE_SPECIALS = ".*+?^${}()|[]\\";

} // namespace


bool IsGlob(TStringBuf str) {
    return str.find('*') != TStringBuf::npos || str.find('?') != TStringBuf::npos;
}

bool IsGlobMatch(TStringBuf glob, TStringBuf str) {
    auto i = 0u, j = 0u;
    for (; i < str.size() && j < glob.size();) {
        if (glob[j] == '?' || glob[j] == str[i]) {
            ++i;
            ++j;
        } else if (glob[j] == '*') {
            ++j;
            if (j >= glob.size()) {
                return true;
            }

            for (; i < str.size(); ++i) {
                if (IsGlobMatch(glob.SubStr(j), str.SubStr(i))) {
                    return true;
                }
            }

            return false;
        } else {
            return false;
        }
    }

    while (j < glob.size() && glob[j] == '*') {
        ++j;
    }

    return j == glob.size() && i == str.size();
}

std::optional<TString> GlobFromRegex(TStringBuf regex) {
    if (regex.empty()) {
        return TString{};
    }

    size_t i = 0, len = regex.size();
    if (regex[i] == '^') {
        // skip first '^'
        i++;
    }
    if (regex[len - 1] == '$') {
        // skip last '$'
        len--;
    }

    TString glob;
    glob.reserve(len);

    // TODO: support conversion of alternatives with and without capturing groups
    // '^(a|b)$'       ->  'a|b'
    // '^(?:a|b)$'     ->  'a|b'
    // '^(a.*b|c.d)$'  ->  'a*b|c?d'

    while (i < len) {
        char ch = regex[i++];

        switch (ch) {
            // unescape special only chars
            case '\\':
                if (i == len) {
                    // hit the end of string, probably invalid regexp
                    return std::nullopt;
                }

                if (RE_SPECIALS.Contains(regex[i])) {
                    glob.append(regex[i]);
                    i++;
                } else {
                    return std::nullopt;
                }
                break;

            // replace
            //     single '.' with '?'
            //     sequence '.*' or '.*?' with '*'
            //     sequence '.+' or '.+?' with '?*'
            case '.':
                if (i == len) {
                    glob.append('?');
                } else if (regex[i] == '*') {
                    glob.append('*');
                    if (++i != len && regex[i] == '?') {
                        ++i; // skip lazy quantifier
                    }
                } else if (regex[i] == '+') {
                    glob.append('?').append('*');
                    if (++i != len && regex[i] == '?') {
                        ++i; // skip lazy quantifier
                    }
                } else {
                    glob.append('?');
                }
                break;

            // copy other non special chars as-is
            default:
                if (RE_SPECIALS.Contains(ch)) {
                    return std::nullopt;
                }
                glob.append(ch);
                break;
        }
    }

    return glob;
}

} // namespace NSolomon
