#include <solomon/libs/cpp/glob/glob.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;

TEST(TGlobTest, IsGlob) {
    EXPECT_TRUE(IsGlob("solomon-*"));
    EXPECT_TRUE(IsGlob("solomon-man0?-01"));
    EXPECT_TRUE(IsGlob("solomon-man0?-01*"));
    EXPECT_TRUE(IsGlob("*"));
    EXPECT_TRUE(!IsGlob("solomon-man-00"));
}

TEST(TGlobTest, MatchQuestion) {
    EXPECT_TRUE(IsGlobMatch("a", "a"));
    EXPECT_TRUE(IsGlobMatch("ab", "ab"));
    EXPECT_TRUE(IsGlobMatch("ab?", "abc"));
    EXPECT_TRUE(IsGlobMatch("a?c", "abc"));
    EXPECT_TRUE(IsGlobMatch("?bc", "abc"));
    EXPECT_TRUE(IsGlobMatch("??c", "abc"));
    EXPECT_TRUE(IsGlobMatch("a??", "abc"));
    EXPECT_TRUE(IsGlobMatch("???", "abc"));
}

TEST(TGlobTest, NotMatchQuestion) {
    EXPECT_TRUE(!IsGlobMatch("", "a"));
    EXPECT_TRUE(!IsGlobMatch("a", ""));
    EXPECT_TRUE(!IsGlobMatch("b", "a"));
    EXPECT_TRUE(!IsGlobMatch("bb?", "abc"));
    EXPECT_TRUE(!IsGlobMatch("?bb", "abc"));
    EXPECT_TRUE(!IsGlobMatch("b?b", "abc"));
    EXPECT_TRUE(!IsGlobMatch("b??", "abc"));
    EXPECT_TRUE(!IsGlobMatch("ab??", "abc"));
    EXPECT_TRUE(!IsGlobMatch("??b", "abc"));
}

TEST(TGlogTest, MatchStart) {
    EXPECT_TRUE(IsGlobMatch("", ""));
    EXPECT_TRUE(IsGlobMatch("*", ""));
    EXPECT_TRUE(IsGlobMatch("**", ""));
    EXPECT_TRUE(IsGlobMatch("a*", "abc"));
    EXPECT_TRUE(IsGlobMatch("ab*", "abc"));
    EXPECT_TRUE(IsGlobMatch("a*c", "abc"));
    EXPECT_TRUE(IsGlobMatch("a*c", "aaaccc"));
    EXPECT_TRUE(IsGlobMatch("*bc", "abc"));
    EXPECT_TRUE(IsGlobMatch("*c", "abc"));
    EXPECT_TRUE(IsGlobMatch("*", "abc"));
    EXPECT_TRUE(IsGlobMatch("*ac", "ac"));
    EXPECT_TRUE(IsGlobMatch("a*c*de", "abcdfbcde"));
    EXPECT_TRUE(IsGlobMatch("a*d*g", "abcdefg"));
    EXPECT_TRUE(IsGlobMatch("a**g", "abcdefg"));
    EXPECT_TRUE(IsGlobMatch("a***", "a"));
    EXPECT_TRUE(IsGlobMatch("a***", "abcdefg"));
    EXPECT_TRUE(IsGlobMatch("**a", "a"));
}

TEST(TGlogTest, NotMatchStart) {
    EXPECT_TRUE(!IsGlobMatch("b*", "abc"));
    EXPECT_TRUE(!IsGlobMatch("bb*", "abc"));
    EXPECT_TRUE(!IsGlobMatch("b*c", "abc"));
    EXPECT_TRUE(!IsGlobMatch("b*c", "bcde"));
    EXPECT_TRUE(!IsGlobMatch("*bb", "abc"));
    EXPECT_TRUE(!IsGlobMatch("*d", "abc"));
    EXPECT_TRUE(!IsGlobMatch("a*z*g", "abcdefg"));
    EXPECT_TRUE(!IsGlobMatch("a*b", "abbe"));
    EXPECT_TRUE(!IsGlobMatch("a*c*g", "abcdefghi"));
}

TEST(TGlobTest, GlobFromRegex) {
    // empty
    {
        auto glob0 = GlobFromRegex("");
        ASSERT_TRUE(glob0.has_value());
        EXPECT_EQ("", *glob0);

        auto glob1 = GlobFromRegex("^$");
        ASSERT_TRUE(glob1.has_value());
        EXPECT_EQ("", *glob1);
    }

    // simple string
    {
        auto glob0 = GlobFromRegex("a");
        ASSERT_TRUE(glob0.has_value());
        EXPECT_EQ("a", *glob0);

        auto glob1 = GlobFromRegex("^a$");
        ASSERT_TRUE(glob1.has_value());
        EXPECT_EQ("a", *glob1);

        auto glob2 = GlobFromRegex("ab");
        ASSERT_TRUE(glob2.has_value());
        EXPECT_EQ("ab", *glob2);

        auto glob3 = GlobFromRegex("^ab$");
        ASSERT_TRUE(glob3.has_value());
        EXPECT_EQ("ab", *glob3);
    }

    // with escaped special chars
    {
        auto glob = GlobFromRegex("foo\\{bar\\}baz");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("foo{bar}baz", *glob);
    }

    // single dot
    {
        auto glob = GlobFromRegex(".");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("?", *glob);
    }

    // multiple dots
    {
        auto glob = GlobFromRegex("..");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("??", *glob);
    }

    // mixed dots
    {
        auto glob0 = GlobFromRegex("a.b.");
        ASSERT_TRUE(glob0.has_value());
        EXPECT_EQ("a?b?", *glob0);

        auto glob1 = GlobFromRegex(".a.b");
        ASSERT_TRUE(glob1.has_value());
        EXPECT_EQ("?a?b", *glob1);

        auto glob2 = GlobFromRegex(".a..b...c.");
        ASSERT_TRUE(glob2.has_value());
        EXPECT_EQ("?a??b???c?", *glob2);
    }

    // dot with '+' quantifier (eager)
    {
        auto glob = GlobFromRegex("a.+b");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("a?*b", *glob);
    }

    // dot with '+' quantifier (lazy)
    {
        auto glob = GlobFromRegex("a.+?b");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("a?*b", *glob);
    }

    // dot with '*' quantifier (eager)
    {
        auto glob = GlobFromRegex("a.*b");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("a*b", *glob);
    }

    // dot with '*' quantifier (lazy)
    {
        auto glob = GlobFromRegex("a.*?b");
        ASSERT_TRUE(glob.has_value());
        EXPECT_EQ("a*b", *glob);
    }

    // dot with '?' quantifier
    {
        auto glob = GlobFromRegex("a.?b");
        ASSERT_FALSE(glob.has_value());
    }


}
