#pragma once

#include <util/stream/file.h>
#include <util/string/strip.h>

#include <solomon/libs/cpp/grpc/interceptor/call_interceptor.h>
#include <solomon/libs/cpp/grpc/interceptor/circuit_breaker_interceptor.h>

namespace NSolomon {
namespace NPrivate {
    const TString USER_AGENT = TString{"YandexSolomon/"} + GetArcadiaLastChange();

    inline grpc::SslCredentialsOptions RetrieveSslOptionsFromConfig(
            const yandex::solomon::config::rpc::TGrpcClientConfig& conf)
    {
        grpc::SslCredentialsOptions sslOptions;

        if (conf.HasSslOptions()) {
            const auto& opts = conf.GetSslOptions();

            if (opts.HasPemRootCertsFilePath()) {
                const auto& filePath = opts.GetPemRootCertsFilePath();
                Y_ENSURE(!filePath.empty(), "PemRootCertsFilePath cannot not be empty");

                TFileInput in{filePath};
                auto pemRootCerts = Strip(in.ReadAll());
                Y_ENSURE(!pemRootCerts.empty(), "pem root certs file cannot be empty");

                sslOptions.pem_root_certs = pemRootCerts;
            } else if (const auto& v = opts.GetPemRootCerts()) {
                sslOptions.pem_root_certs = v;
            }
        }

        return sslOptions;
    }

    inline std::unique_ptr<grpc::experimental::ClientInterceptorFactoryInterface> CreateClientCircuitBreaker(
            NMonitoring::IMetricRegistry& registry,
            const yandex::solomon::config::rpc::TCircuitBreakerConfig& conf)
    {
        return ClientCircuitBreakerInterceptorFactory(
                registry,
                100 * conf.GetFailureQuantileThreshold(),
                FromProtoTime(conf.GetResetTimeout(), TDuration::Minutes(1)));
    }

    inline std::shared_ptr<grpc::ChannelInterface> CreateChannelInterface(
        const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
        const TString& address,
        bool secure,
        NMonitoring::IMetricRegistry& registry,
        TString clientId)
    {
        TVector<TInterceptorFactoryPtr> interceptors;
        interceptors.push_back(CreateCounterInterceptorFactory(registry, std::move(clientId)));
        if (conf.HasCircuitBreakerConfig()) {
            interceptors.push_back(NPrivate::CreateClientCircuitBreaker(registry, conf.GetCircuitBreakerConfig()));
        }

        grpc::ChannelArguments args;
        if (conf.HasMaxInboundMessageSize()) {
            args.SetMaxReceiveMessageSize(FromProtoDataSize(conf.GetMaxInboundMessageSize()));
        }

        if (conf.HasMaxOutboundMessageSize()) {
            args.SetMaxSendMessageSize(FromProtoDataSize(conf.GetMaxOutboundMessageSize()));
        }

        if (conf.HasMemoryQuota()) {
            grpc::ResourceQuota quota;
            quota.Resize(FromProtoDataSize(conf.GetMemoryQuota()));
            args.SetResourceQuota(quota);
        }

        if (conf.HasIdleTimeout()) {
            args.SetInt(GRPC_ARG_CLIENT_IDLE_TIMEOUT_MS, FromProtoTime(conf.GetIdleTimeout()).MilliSeconds());
        }

        if (conf.HasKeepAliveTime()) {
            args.SetInt(GRPC_ARG_KEEPALIVE_TIME_MS, FromProtoTime(conf.GetKeepAliveTime()).MilliSeconds());
        }

        if (conf.HasKeepAliveTimeout()) {
            args.SetInt(GRPC_ARG_KEEPALIVE_TIMEOUT_MS, FromProtoTime(conf.GetKeepAliveTimeout()).MilliSeconds());
        }

        if (secure) {
            grpc::SslCredentialsOptions sslOptions = RetrieveSslOptionsFromConfig(conf);
            auto sslCreds = grpc::SslCredentials(sslOptions);
            return grpc::experimental::CreateCustomChannelWithInterceptors(address, sslCreds, args, std::move(interceptors));
        }

        return grpc::experimental::CreateCustomChannelWithInterceptors(address, grpc::InsecureChannelCredentials(), args, std::move(interceptors));
    }
} // namespace NPrivate
} // namespace NSolomon
