#pragma once

#include "error.h"

#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/error_or/error_or.h>
#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/protos/configs/rpc/rpc_config.pb.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>
#include <library/cpp/grpc/client/grpc_client_low.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/string_utils/url/url.h>
#include <library/cpp/svnversion/svnversion.h>
#include <library/cpp/threading/future/future.h>

#include <util/datetime/base.h>
#include <util/generic/hash_set.h>
#include <util/generic/string.h>
#include <util/random/random.h>
#include <util/string/builder.h>
#include <util/string/cast.h>

#include "client-inl.h"

namespace NSolomon {
    template <typename T>
    TErrorOr<T, TApiCallError> FromGrpcError(::NGrpc::TGrpcStatus status) {
        return TErrorOr<T, TApiCallError>::FromError(
            std::move(status),
            TStringBuilder() << "gRPC error (" << status.GRpcStatusCode << "):" << status.Msg
        );
    }

    template <typename TService>
    class TGrpcServiceConnection {
    public:
        TGrpcServiceConnection(
                TDuration timeout,
                std::unique_ptr<::NGrpc::TServiceConnection<TService>> connection,
                std::unique_ptr<::NGrpc::TStubsHolder> stubs,
                // passed only for a single host client
                std::unique_ptr<::NGrpc::TGRpcClientLow> threadPool = {})
            : Timeout_{timeout}
            , Connection_{std::move(connection)}
            , Stubs_{std::move(stubs)}
            , ThreadPool_{std::move(threadPool)}
        {
        }

        template<typename TRequest, typename TResponse>
        void Request(
                const TRequest& request,
                ::NGrpc::TResponseCallback<TResponse>&& callback,
                typename ::NGrpc::TSimpleRequestProcessor<typename TService::Stub, TRequest, TResponse>::TAsyncRequest asyncRequest)
        {
            Connection_->DoRequest(
                request,
                std::move(callback),
                asyncRequest,
                ::NGrpc::TCallMeta{
                    .Timeout = Timeout_,
                });
        }

        template<typename TRequest, typename TResponse>
        void Request(
                const TRequest& request,
                ::NGrpc::TResponseCallback<TResponse>&& callback,
                typename ::NGrpc::TSimpleRequestProcessor<typename TService::Stub, TRequest, TResponse>::TAsyncRequest asyncRequest,
                ::NGrpc::TCallMeta meta)
        {
            Connection_->DoRequest(
                request,
                std::move(callback),
                asyncRequest,
                std::move(meta));
        }

        TDuration Timeout() const noexcept {
            return Timeout_;
        }

    private:
        TDuration Timeout_;
        std::unique_ptr<::NGrpc::TServiceConnection<TService>> Connection_;
        /**
         * These pointers are stored to own objects associated with a connection.
         * When this connection is destructed, refcount will become 0, hence stubs and threadPool will be freed
         */
        std::unique_ptr<::NGrpc::TStubsHolder> Stubs_;
        std::unique_ptr<::NGrpc::TGRpcClientLow> ThreadPool_;
    };

    inline std::unique_ptr<::NGrpc::TGRpcClientLow> CreateGrpcThreadPool(
            const yandex::solomon::config::rpc::TGrpcClientConfig& conf)
    {
        const auto numThreads = conf.HasWorkerThreads()
            ? conf.GetWorkerThreads()
            : ::NGrpc::DEFAULT_NUM_THREADS;
        return std::make_unique<::NGrpc::TGRpcClientLow>(numThreads, true);
    }

    /**
     * Creates a service connection owning a threadPool
     */
    template <typename TService>
    inline std::unique_ptr<TGrpcServiceConnection<TService>> CreateGrpcServiceConnection(
            const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
            bool secure,
            NMonitoring::IMetricRegistry& registry,
            std::unique_ptr<::NGrpc::TGRpcClientLow>&& threadPool,
            TString clientId = {})
    {
        const auto& address = conf.GetAddresses(0);
        Y_ENSURE(!address.empty(), "gRPC address must be specified");

        auto stubs = std::make_unique<::NGrpc::TStubsHolder>(
            NPrivate::CreateChannelInterface(conf, address, secure, registry, std::move(clientId)));
        auto conn = threadPool->CreateGRpcServiceConnection<TService>(*stubs);
        auto timeout = conf.HasReadTimeout() ? FromProtoTime(conf.GetReadTimeout()) : TDuration::Zero();

        return std::make_unique<TGrpcServiceConnection<TService>>(
            timeout,
            std::move(conn),
            std::move(stubs),
            std::move(threadPool));
    }

    /**
     * Creates a service connection not owning a threadPool
     */
    template <typename TService>
    inline std::unique_ptr<TGrpcServiceConnection<TService>> CreateGrpcServiceConnection(
            const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
            const TString& address,
            bool secure,
            NMonitoring::IMetricRegistry& registry,
            ::NGrpc::TGRpcClientLow& threadPool,
            TString clientId = {})
    {
        Y_ENSURE(!address.empty(), "gRPC address must be specified");

        auto stubs = std::make_unique<::NGrpc::TStubsHolder>(
            NPrivate::CreateChannelInterface(conf, address, secure, registry, std::move(clientId)));
        auto conn = threadPool.CreateGRpcServiceConnection<TService>(*stubs);
        auto timeout = conf.HasReadTimeout() ? FromProtoTime(conf.GetReadTimeout()) : TDuration::Zero();

        return std::make_unique<TGrpcServiceConnection<TService>>(
            timeout,
            std::move(conn),
            std::move(stubs));
    }

    template <typename TClientImpl>
    struct IClusterRpc {
        virtual ~IClusterRpc() = default;

        virtual void Stop(bool wait) = 0;
        virtual void Add(TStringBuf address) = 0;
        virtual TClientImpl* Get(TStringBuf address) noexcept = 0;
        virtual TClientImpl* GetAny() noexcept = 0;
        virtual TString GetAnyAddress() const noexcept = 0;
        virtual const std::vector<TString>& Addresses() const noexcept = 0;
    };

    template <typename TService, typename TClientImpl>
    class TGrpcClusterClientBase final: public IClusterRpc<TClientImpl>, TMoveOnly {
        using TClientFactory = std::function<
            std::unique_ptr<TClientImpl>(NMonitoring::IMetricRegistry&, std::unique_ptr<TGrpcServiceConnection<TService>>)>;

    public:
        TGrpcClusterClientBase(
                const yandex::solomon::config::rpc::TGrpcClientConfig& conf,
                bool secure,
                NMonitoring::IMetricRegistry& registry,
                TClientFactory clientFactory,
                TString clientId = {})
            : Config_{conf}
            , IsConnectionSecure_{secure}
            , Registry_{registry}
            , ClientFactory_{clientFactory}
            , ClientId_{std::move(clientId)}
            , ThreadPool_{CreateGrpcThreadPool(conf)}
        {
            Clients_.reserve(conf.AddressesSize());
            Addresses_.reserve(conf.AddressesSize());

            for (auto& addr: conf.GetAddresses()) {
                Add(addr);
            }
        }

        ~TGrpcClusterClientBase() {
            ThreadPool_->Stop(true);
        }

        void Stop(bool wait) override {
            ThreadPool_->Stop(wait);
        }

        void Add(TStringBuf address) override {
            if (Clients_.find(address) != Clients_.end()) {
                return;
            }

            auto serviceConnection = CreateGrpcServiceConnection<TService>(
                Config_,
                TString{address},
                IsConnectionSecure_,
                Registry_,
                *ThreadPool_,
                ClientId_);

            auto hostAndPort = ToString(GetHostAndPort(address));
            Clients_.emplace(hostAndPort, ClientFactory_(Registry_, std::move(serviceConnection)));
            Addresses_.emplace_back(hostAndPort);
        }

        TClientImpl* Get(TStringBuf address) noexcept override {
            if (auto it = Clients_.find(address); it != Clients_.end()) {
                return it->second.get();
            }

            return nullptr;
        }

        TClientImpl* GetAny() noexcept override {
            if (Clients_.empty()) {
                return nullptr;
            }

            return std::next(Clients_.begin(), RandomNumber<ui64>(Clients_.size()))->second.get();
        }

        TString GetAnyAddress() const noexcept override {
            if (Addresses_.empty()) {
                return {};
            }

            auto rnd = RandomNumber<size_t>(Addresses_.size());
            return Addresses_[rnd];
        }

        const std::vector<TString>& Addresses() const noexcept override {
            return Addresses_;
        };

    private:
        const yandex::solomon::config::rpc::TGrpcClientConfig Config_;
        bool IsConnectionSecure_{false};
        NMonitoring::IMetricRegistry& Registry_;
        TClientFactory ClientFactory_;
        TString ClientId_;
        std::unique_ptr<::NGrpc::TGRpcClientLow> ThreadPool_;
        TStringMap<std::unique_ptr<TClientImpl>> Clients_;
        TVector<TString> Addresses_;
    };
} // namespace NSolomon
