#pragma once

#include <solomon/libs/cpp/sync/rw_lock.h>

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/system/hp_timer.h>

#include <grpcpp/grpcpp.h>

namespace NSolomon {
    class ICallCounters {
    public:
        virtual ~ICallCounters() = default;
        virtual void ReportCallStart() = 0;
        virtual void ReportCallStats(grpc::StatusCode code, TDuration duration) = 0;
        virtual void ReportDelivery(ui64 durationMillis) = 0;
        virtual void ReportDelivery(TDuration duration) = 0;
        virtual void ReportInboundBytes(ui64 bytes) = 0;
        virtual void ReportOutboundBytes(ui64 bytes) = 0;
    };

    struct TBaseCallCounters: public ICallCounters {
    public:
        TBaseCallCounters(NMonitoring::IMetricRegistry& registry, const NMonitoring::TLabels& labels, const TString& prefix)
            : Registry_{registry}
            , CommonLabels_{labels}
            , Prefix_{prefix}
        {
            Timings_ = registry.HistogramRate(
                MakeLabels(".call.elapsedTimeMs"),
                NMonitoring::ExponentialHistogram(13, 2, 16));

            Delivery_ = registry.HistogramRate(
                MakeLabels(".call.deliveryTimeMs"),
                NMonitoring::ExponentialHistogram(16, 2, 1));

            Started_ = registry.Rate(MakeLabels(".call.started"));
            Completed_ = registry.Rate(MakeLabels(".call.completed"));
            InboundBytes_ = registry.Rate(MakeLabels(".call.inBoundBytes"));
            OutboundBytes_ = registry.Rate(MakeLabels(".call.outBoundBytes"));
            Inflight_ = registry.IntGauge(MakeLabels(".call.inFlight"));
        }
        virtual ~TBaseCallCounters() = default;

        void ReportCallStart() override;
        void ReportCallStats(grpc::StatusCode code, TDuration duration) override;
        void ReportDelivery(ui64 durationMillis) override;
        void ReportDelivery(TDuration duration) override;
        void ReportInboundBytes(ui64 bytes) override;
        void ReportOutboundBytes(ui64 bytes) override;
    private:
        NMonitoring::IMetricRegistry& Registry_;
        const NMonitoring::TLabels CommonLabels_;
        const TString  Prefix_;
        NMonitoring::IHistogram* Delivery_{};
        NMonitoring::IHistogram* Timings_{};
        NMonitoring::IRate* Started_{};
        NMonitoring::IRate* Completed_{};
        NMonitoring::IRate* InboundBytes_{};
        NMonitoring::IRate* OutboundBytes_{};
        NMonitoring::IIntGauge* Inflight_{};
        NSync::TLightRwLock<THashMap<grpc::StatusCode, NMonitoring::IRate*>> Status_;

        NMonitoring::ILabelsPtr MakeLabels(TStringBuf name, const NMonitoring::TLabels& additional = {});
        NMonitoring::IRate* ResolveStatus(grpc::StatusCode code);
    };

    struct TServerCallCounters: public TBaseCallCounters {
        TServerCallCounters(NMonitoring::IMetricRegistry& registry, const NMonitoring::TLabels& labels)
            : TBaseCallCounters(registry, labels, "grpc.server")
        {
        };
    };

    struct TClientCallCounters: public TBaseCallCounters {
        TClientCallCounters(NMonitoring::IMetricRegistry& registry, const NMonitoring::TLabels& labels)
            : TBaseCallCounters(registry, labels, "grpc.client")
        {
        };
    };

    struct TAggregateCallCounters: public ICallCounters {
    public:
        TAggregateCallCounters(TBaseCallCounters* target, TBaseCallCounters* total)
            : Target_(target)
            , Total_(total)
        {
        }

        void ReportCallStart() override {
            Target_->ReportCallStart();
            Total_->ReportCallStart();
        }

        void ReportCallStats(grpc::StatusCode code, TDuration duration) override {
            Target_->ReportCallStats(code, duration);
            Total_->ReportCallStats(code, duration);
        }

        void ReportDelivery(ui64 durationMillis) override {
            Target_->ReportDelivery(durationMillis);
            Total_->ReportDelivery(durationMillis);
        }

        void ReportDelivery(TDuration duration) override {
            ReportDelivery(duration.MilliSeconds());
        }

        void ReportInboundBytes(ui64 bytes) override {
            Target_->ReportInboundBytes(bytes);
            Total_->ReportInboundBytes(bytes);
        }

        void ReportOutboundBytes(ui64 bytes) override {
            Target_->ReportOutboundBytes(bytes);
            Total_->ReportOutboundBytes(bytes);
        }

    private:
        TBaseCallCounters* Target_;
        TBaseCallCounters* Total_;
    };
} // namespace NSolomon
