#include <solomon/libs/cpp/grpc/metrics/counters.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;

TEST(TGrpcCountersTest, ClientEmptyLabels) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TClientCallCounters(registry, NMonitoring::TLabels());

    auto started = registry.Rate({{"sensor", "grpc.client.call.started"}});
    auto completed = registry.Rate({{"sensor", "grpc.client.call.completed"}});
    auto inFlight = registry.IntGauge({{"sensor", "grpc.client.call.inFlight"}});

    EXPECT_EQ(0u, started->Get());
    EXPECT_EQ(0u, completed->Get());
    EXPECT_EQ(0, inFlight->Get());

    metrics.ReportCallStart();
    EXPECT_EQ(1u, started->Get());
    EXPECT_EQ(0u, completed->Get());
    EXPECT_EQ(1, inFlight->Get());

    metrics.ReportCallStart();
    EXPECT_EQ(2u, started->Get());
    EXPECT_EQ(0u, completed->Get());
    EXPECT_EQ(2, inFlight->Get());

    metrics.ReportCallStats(grpc::StatusCode::OK, TDuration::MilliSeconds(10));
    EXPECT_EQ(2u, started->Get());
    EXPECT_EQ(1u, completed->Get());
    EXPECT_EQ(1, inFlight->Get());

    metrics.ReportCallStats(grpc::StatusCode::INTERNAL, TDuration::MilliSeconds(5));
    EXPECT_EQ(2u, started->Get());
    EXPECT_EQ(2u, completed->Get());
    EXPECT_EQ(0, inFlight->Get());
}

TEST(TGrpcCountersTest, ClientLables) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TClientCallCounters(registry, {{"myKey", "myValue"}});

    auto started = registry.Rate({{"sensor", "grpc.client.call.started"}, {"myKey", "myValue"}});
    auto inFlight = registry.IntGauge({{"sensor", "grpc.client.call.inFlight"}, {"myKey", "myValue"}});
    for (int index = 1; index <= 10; index++) {
        metrics.ReportCallStart();
        EXPECT_EQ(static_cast<ui64>(index), started->Get());
        EXPECT_EQ(index, inFlight->Get());
    }
}

TEST(TGrpcCountersTest, ServerLables) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TServerCallCounters(registry, {{"myKey", "myValue"}});

    auto started = registry.Rate({{"sensor", "grpc.server.call.started"}, {"myKey", "myValue"}});
    auto inFlight = registry.IntGauge({{"sensor", "grpc.server.call.inFlight"}, {"myKey", "myValue"}});
    for (int index = 1; index <= 10; index++) {
        metrics.ReportCallStart();
        EXPECT_EQ(static_cast<ui64>(index), started->Get());
        EXPECT_EQ(index, inFlight->Get());
    }

    auto completed = registry.Rate({{"sensor", "grpc.server.call.completed"}, {"myKey", "myValue"}});
    metrics.ReportCallStats(grpc::StatusCode::INTERNAL, TDuration::MilliSeconds(5));
    EXPECT_EQ(1u, completed->Get());
    EXPECT_EQ(9, inFlight->Get());
}

TEST(TGrpcCountersTest, Aggregate) {
    NMonitoring::TMetricRegistry registry;
    auto alice = TServerCallCounters(registry, {{"target", "alice"}});
    auto bob = TServerCallCounters(registry, {{"target", "bob"}});
    auto aggr = TServerCallCounters(registry, {{"target", "total"}});

    {   // 2 started, 1 completed
        auto metrics = TAggregateCallCounters(&alice, &aggr);
        metrics.ReportCallStart();
        metrics.ReportCallStart();
        metrics.ReportCallStats(grpc::StatusCode::OK, TDuration::MilliSeconds(5));
    }

    {   // 3 started, 1 completed
        auto metrics = TAggregateCallCounters(&bob, &aggr);
        metrics.ReportCallStart();
        metrics.ReportCallStats(grpc::StatusCode::OK, TDuration::MilliSeconds(5));
        metrics.ReportCallStart();
        metrics.ReportCallStart();
    }

    EXPECT_EQ(2u, registry.Rate({{"sensor", "grpc.server.call.started"}, {"target", "alice"}})->Get());
    EXPECT_EQ(3u, registry.Rate({{"sensor", "grpc.server.call.started"}, {"target", "bob"}})->Get());
    EXPECT_EQ(5u, registry.Rate({{"sensor", "grpc.server.call.started"}, {"target", "total"}})->Get());

    EXPECT_EQ(1u, registry.Rate({{"sensor", "grpc.server.call.completed"}, {"target", "alice"}})->Get());
    EXPECT_EQ(1u, registry.Rate({{"sensor", "grpc.server.call.completed"}, {"target", "bob"}})->Get());
    EXPECT_EQ(2u, registry.Rate({{"sensor", "grpc.server.call.completed"}, {"target", "total"}})->Get());

    EXPECT_EQ(1, registry.IntGauge({{"sensor", "grpc.server.call.inFlight"}, {"target", "alice"}})->Get());
    EXPECT_EQ(2, registry.IntGauge({{"sensor", "grpc.server.call.inFlight"}, {"target", "bob"}})->Get());
    EXPECT_EQ(3, registry.IntGauge({{"sensor", "grpc.server.call.inFlight"}, {"target", "total"}})->Get());
}

TEST(TGrpcCountersTest, StatusCount) {
    NMonitoring::TMetricRegistry registry;
    auto metrics = TClientCallCounters(registry, NMonitoring::TLabels());

    metrics.ReportCallStart();
    metrics.ReportCallStats(grpc::StatusCode::OK, TDuration::MilliSeconds(10));

    auto ok = registry.Rate({{"sensor", "grpc.client.call.status"}, {"code", "OK"}});
    auto deadline = registry.Rate({{"sensor", "grpc.client.call.status"}, {"code", "DEADLINE_EXCEEDED"}});
    EXPECT_EQ(1u, ok->Get());
    EXPECT_EQ(0u, deadline->Get());

    metrics.ReportCallStart();
    metrics.ReportCallStats(grpc::StatusCode::OK, TDuration::MilliSeconds(10));
    EXPECT_EQ(2u, ok->Get());
    EXPECT_EQ(0u, deadline->Get());

    metrics.ReportCallStart();
    metrics.ReportCallStats(grpc::StatusCode::DEADLINE_EXCEEDED, TDuration::MilliSeconds(10));
    EXPECT_EQ(2u, ok->Get());
    EXPECT_EQ(1u, deadline->Get());
}
