#pragma once

#include "events.h"
#include "request_state.h"

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

namespace NSolomon::NGrpc {

template <typename T>
concept HasHandleEvent = requires(T t, STFUNC_SIG) {
    t.HandleEvent(ev, ctx);
};

template <typename TDerive>
class TBaseHandler: public NActors::TActorBootstrapped<TBaseHandler<TDerive>> {
    static constexpr auto TimerDelay = TDuration::Seconds(2);
    static constexpr auto MaxTimeout = TDuration::Minutes(5);

protected:
    enum class EMode {
        Sync,
        Async,
    };

public:
    TBaseHandler()
        : Timer_{TInstant::Now()}
    {
    }

    void Bootstrap() {
        this->Schedule(TimerDelay, new NActors::TEvents::TEvWakeup);
        this->Become(&TBaseHandler<TDerive>::StateFunc);
    }

private:
    // derived class should not switch actor's state function
    using NActors::TActorBootstrapped<TBaseHandler<TDerive>>::Become;

    STFUNC(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            HFunc(TServerEvents::TRequest, OnRequest);
            hFunc(TServerEvents::TResponse, OnResponse);
            hFunc(TServerEvents::TError, OnError);
            sFunc(NActors::TEvents::TEvWakeup, OnWakeup);
            hFunc(NActors::TEvents::TEvPoison, OnPoison);

            default: {
                // all other events are send to derive class iff it has appropriate method
                auto& self = static_cast<TDerive&>(*this);
                if constexpr(HasHandleEvent<TDerive>) {
                    self.HandleEvent(ev, ctx);
                }
            }
        }
    }

    void OnRequest(TServerEvents::TRequest::TPtr& ev, const NActors::TActorContext& ctx) {
        auto* reqCtx = ev->Get()->ReqCtx;
        auto receivedAt = ev->Get()->ReceivedAt;

        auto reqDeadline = reqCtx->Deadline();
        auto nowUpperBound = Timer_.Now() + Timer_.TickDuration();
        if (reqDeadline <= nowUpperBound) {
            // request already expired, reply with DEADLINE_EXCEEDED
            TStringBuilder sb;
            sb << "request already expired, deadline (" << reqDeadline << ") <= now (" << nowUpperBound << ')';
            reqCtx->ReplyError(grpc::DEADLINE_EXCEEDED, sb);
            return;
        }

        auto reqState = std::make_unique<TRequestState>(reqCtx, receivedAt);
        auto& self = static_cast<TDerive&>(*this);

        if (self.HandleRequest(reqState.get()) == EMode::Async) {
            // timeouts scheduled only if request handled asynchronously
            auto deadline = Min(reqDeadline, ctx.Now() + MaxTimeout);
            reqState->ScheduleDeadline(deadline, Timer_);
            InFlight_.PushBack(reqState.release());
        }
    }

    void OnResponse(TServerEvents::TResponse::TPtr& ev) {
        auto req = TRequestState::FromCookie(ev->Cookie);
        req->SendReply(ev->Get()->Message.get(), ev->Get()->Code);
    }

    void OnError(TServerEvents::TError::TPtr& ev) {
        auto req = TRequestState::FromCookie(ev->Cookie);
        req->SendError(ev->Get()->Code, ev->Get()->Message);
    }

    void OnWakeup() {
        auto elapsed = NActors::TActivationContext::Now() - Timer_.Now();
        if (elapsed >= Timer_.TickDuration()) {
            Timer_.Advance(elapsed);
        }
        this->Schedule(TimerDelay, new NActors::TEvents::TEvWakeup);
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        while (!InFlight_.Empty()) {
            std::unique_ptr<TRequestState> req{InFlight_.PopFront()};
            req->SendError(grpc::ABORTED, "server is shutting down");
        }

        this->Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        this->PassAway();
    }

private:
    NTimer::TTimerWheel Timer_;
    TIntrusiveList<TRequestState> InFlight_;
};

} // namespace NSolomon::NGrpc
