#pragma once

#include <solomon/libs/cpp/threading/timer/timer_wheel.h>

#include <library/cpp/grpc/server/grpc_request_base.h>
#include <library/cpp/actors/core/actor.h>

#include <util/string/builder.h>

namespace NSolomon::NGrpc {

class TRequestState: public TIntrusiveListItem<TRequestState> {
public:
    TRequestState(::NGrpc::IRequestContextBase* reqCtx, TInstant receivedAt) noexcept
        : ReqCtx_{reqCtx}
        , ReceivedAt_{receivedAt}
        , Timeout_{this}
    {
    }

    ~TRequestState() noexcept {
        Y_VERIFY(IsReplied(), "deleting request without replying to client");

        // remove from inflight requests list
        this->Unlink();

        // cancel timeout
        Timeout_.Cancel();
    }

    template <typename TMessage>
    const TMessage* GetMessage() {
        Y_VERIFY(ReqCtx_, "trying to access request to which response has already been sent");
        static_assert(std::is_base_of_v<NProtoBuf::Message, TMessage>, "incorrect message type");
        return static_cast<const TMessage*>(ReqCtx_->GetRequest());
    }

    google::protobuf::Arena* Arena() {
        Y_VERIFY(ReqCtx_, "trying to access request to which response has already been sent");
        return ReqCtx_->GetArena();
    }

    void SendReply(NProtoBuf::Message* message, grpc::StatusCode code = grpc::OK) {
        if (!IsReplied()) {
            ReqCtx_->Reply(message, static_cast<ui32>(code));
            ReqCtx_ = nullptr;
        }
    }

    void SendError(grpc::StatusCode code, const TString& message) {
        if (!IsReplied()) {
            ReqCtx_->ReplyError(code, message);
            ReqCtx_ = nullptr;
        }
    }

    bool IsReplied() const noexcept {
        return ReqCtx_ == nullptr;
    }

    void ScheduleDeadline(TInstant deadline, NTimer::TTimerWheel& timer) {
        Y_VERIFY(!Timeout_.IsScheduled(), "request deadline is already scheduled");
        timer.ScheduleAt(&Timeout_, deadline);
    }

    ui64 ToCookie() const noexcept {
        return reinterpret_cast<ui64>(this);
    }

    static std::unique_ptr<TRequestState> FromCookie(ui64 cookie) noexcept {
        return std::unique_ptr<TRequestState>{reinterpret_cast<TRequestState*>(cookie)};
    }

private:
    void OnTimeout() {
        TStringBuilder sb;
        sb << "request timeout after " << (NActors::TActivationContext::Now() - ReceivedAt_);
        SendError(grpc::DEADLINE_EXCEEDED, sb);

        // we do not drop this object on timeout, because there is some running operation
        // which has a raw reference to this object
    }

private:
    ::NGrpc::IRequestContextBase* ReqCtx_;
    const TInstant ReceivedAt_;
    NTimer::TMemFuncEvent<TRequestState, &TRequestState::OnTimeout> Timeout_;
};

} // namespace NSolomon::NGrpc
