#pragma once

#include "events.h"

#include <solomon/libs/cpp/actors/config/log_component.pb.h>

#include <library/cpp/grpc/server/grpc_request.h>
#include <library/cpp/grpc/server/grpc_server.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/defs.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <chrono>

namespace NSolomon::NGrpc {

namespace NPrivate {

struct TActorSystemMetrics {
    NMonitoring::IIntGauge* Queue = nullptr;
    NMonitoring::IRate* SendRate = nullptr;
    NMonitoring::IRate* ReceiveRate = nullptr;
    NMonitoring::IHistogram* DeliveryTime = nullptr;

    TActorSystemMetrics() = default;

    TActorSystemMetrics(NMonitoring::TMetricRegistry& reg, TString endpoint)
        : Queue(reg.IntGauge(
            {{"sensor", "grpc.server.actorEvents.queue"}, {"endpoint", endpoint}}))
        , SendRate(reg.Rate(
            {{"sensor", "grpc.server.actorEvents.send"}, {"endpoint", endpoint}}))
        , ReceiveRate(reg.Rate(
            {{"sensor", "grpc.server.actorEvents.receive"}, {"endpoint", endpoint}}))
        , DeliveryTime(reg.HistogramRate(
            {{"sensor", "grpc.server.actorEvents.deliveryTimeUs"}, {"endpoint", endpoint}},
            NMonitoring::ExponentialHistogram(20, 2, 4)))
    {
    }

    void OnCallRegistered() {
        if (Queue) Queue->Inc();
        if (SendRate) SendRate->Inc();
    }

    void OnCallProcessingStarted(ui64 queueTimeUs) {
        if (Queue) Queue->Dec();
        if (ReceiveRate) ReceiveRate->Inc();
        if (DeliveryTime) DeliveryTime->Record(queueTimeUs);
    }
};

// We can't use `THandler` in `std::is_base_of` because it is a class template.
// So we use this tag to check inheritance instead.
struct THandlerTag {};

// Deduce input and output message types from callback signature.
// @{

// No match -- this is not an async callback. Are you sure you've set `THandler::Callback` correctly?
template <typename T>
struct TSignature {
    static_assert(!std::is_same_v<T, T>, "not an async callback");
};

template <typename TService, typename TRequest, typename TResponse>
struct TSignature<
    void (TService::*)(grpc::ServerContext*, TRequest*, grpc::ServerAsyncResponseWriter<TResponse>*, grpc::CompletionQueue*, grpc::ServerCompletionQueue*, void*)
> {
    using Request = TRequest;
    using Response = TResponse;
};

// @}

} // namespace NPrivate

inline constexpr TDuration MAX_API_TIMEOUT = TDuration::Minutes(5);

/// Base class for all rpc request handlers.
///
/// This is a standard actor. For each request, a new instance of this actor is created and registered.
///
/// Upon registration, the `Handle` method is called to process the request.
///
/// If processing takes too much time, `NActors::TEvents::TEvWakeup` will be sent to this actor indicating that results
/// are no longer needed. Call `SendTimeoutAndPassAway` when it happens.
///
/// @tparam TDerived
///     any handler implementation must pass itself as a template parameter -- see [CRTP] for more info.
///
/// [CRTP]: https://en.wikipedia.org/wiki/Curiously_recurring_template_pattern
///
///
/// # Usage
///
/// Note: prefer using the `SOLOMON_RPC_HANDLER` macro to create implementations of this interface.
///
/// Derive a handler from this class. In handler body, define three type aliases (`TGeneratedService`,
/// `TRequest`, `TResponse`) and two static constexpr const variables (`Name`, `Callback`). Note,
/// header must be final.
///
/// Suppose we have the following protobuf:
///
/// ```
/// syntax = "proto3";
///
/// package generated;
///
/// service GreetingService {
///     rpc Greet (TGreetRequest) returns (TGreetResponse) {}
/// }
///
/// message GreetRequest {
///     string Name = 1;
/// }
///
/// message GreetResponse {
///     string Greeting = 1;
/// }
/// ```
///
/// Then we can implement the following handler:
///
/// ```
/// class TGreetHandler final: public ::NSolomon::NGrpc::THandler<TGreetHandler> {
/// public:
///     using TGeneratedService = generated::GreetingService;
///     using TRequest = generated::GreetRequest;
///     using TResponse = generated::GreetResponse;
///     static constexpr TStringBuf Name = "Greet";
///     static constexpr const auto Callback = &TGeneratedService::AsyncService::RequestGreet;
///
///     void Handle(const NActors::TActorContext& ctx) override {
///         auto response = NProtos::TTestResponse();
///         response.SetGreeting("Hello, " + GetRequest()->GetName());
///         GetRequestCtx()->Reply(&response);
///         PassAway();
///     }
/// };
/// ```
template <typename TDerived>
class THandler: public NActors::TActorBootstrapped<TDerived>, public NPrivate::THandlerTag {
    template <typename TGeneratedService>
    friend class TService;

private:
    /**
     * Calling Become from a derived class is prohibited and only this bacse class can handle an incoming event.
     * If you want to write your own handlers, implement a STATEFN(StateFunc) inside a derived class.
     * For more info, see STFUNC(State) inside this class
     */
    using NActors::TActorBootstrapped<TDerived>::Become;

public:
    /// Service class as generated by protoc, must be shadowed in derived class.
    using TGeneratedService = void;

    /// Proto message that's used as remote function parameter type, must be shadowed in derived class.
    using TRequest = void;

    /// Proto message that's used as remote function return type, must be shadowed in derived class.
    using TResponse = void;

    /// Method name, must be shadowed in derived class.
    /// This name is only used in type checking and logging.
    /// We can't properly check that `Name` matches the `Callback` implementation.
    static constexpr TStringBuf Name = TStringBuf();

    /// A pointer to the callback member function of the `TGeneratedService::AsyncService` class that
    /// dispatches requests for this handler.
    static constexpr const void* Callback = nullptr;

public:
    THandler() {
        static_assert(
            sizeof(TDerived) != 0,
            "trying to instantiate 'THandler' with an incomplete type parameter");
        static_assert(
            std::is_base_of_v<THandler<TDerived>, TDerived>,
            "handler implementations must pass itself as 'TDerived' parameter");
        static_assert(
            std::is_final_v<TDerived>,
            "handler implementations must be marked 'final'");
        static_assert(
            !std::is_same_v<typename TDerived::TGeneratedService, void>,
            "'TGeneratedService' must be overridden in handler implementation");
        static_assert(
            !std::is_same_v<typename TDerived::TRequest, void>,
            "'TRequest' must be overridden in handler implementation");
        static_assert(
            std::is_base_of_v<google::protobuf::Message, typename TDerived::TRequest>,
            "'TRequest' must be a protobuf message");
        static_assert(
            !std::is_same_v<typename TDerived::TResponse, void>,
            "'TResponse' must be overridden in handler implementation");
        static_assert(
            std::is_base_of_v<google::protobuf::Message, typename TDerived::TResponse>,
            "'TResponse' must be a protobuf message");
        static_assert(
            std::is_same_v<std::remove_const_t<decltype(TDerived::Name)>, TStringBuf>,
            "'Name' must be a 'TStringBuf'");
        static_assert(
            TDerived::Name.Size() >= 0,
            "'Name' must be overridden with non-empty string in handler implementation");
    }

    void Bootstrap(const NActors::TActorContext& ctx) {
        auto now = std::chrono::steady_clock::now();
        auto queueTimeUs = std::chrono::duration_cast<std::chrono::microseconds>(CreateTime_ - now).count();

        Endpoint_.OnCallProcessingStarted(queueTimeUs);
        Total_.OnCallProcessingStarted(queueTimeUs);

        NActors::TActorBootstrapped<TDerived>::Become(&TDerived::State);
        NActors::TActorBootstrapped<TDerived>::Schedule(GetRequestTimeout(), new NActors::TEvents::TEvWakeup{});

        // downcast to `TDerived` so that compiler can eliminate this virtual call -- `TDerived` is `final`.
        static_cast<TDerived*>(this)->Handle(ctx);
    }

    STFUNC(State) {
        switch (ev->GetTypeRewrite()) {
            sFunc(NActors::TEvents::TEvWakeup, SendTimeoutAndPassAway);

            default:
                return static_cast<TDerived*>(this)->StateFunc(ev, ctx);
        }
    }

public:
    /// Actual implementation of request processing logic.
    virtual void Handle(const NActors::TActorContext& ctx) = 0;

protected:
    /// Get context for the current request.
    auto GetRequestCtx() const {
        return RequestCtx_;
    }

    /// Get protobuf message with request parameters.
    auto GetRequest() {
        return CheckedCast<const typename TDerived::TRequest*>(RequestCtx_->GetRequest());
    }

    /// Get timeout for this request.
    auto GetRequestTimeout() const {
        auto deadline = RequestCtx_->Deadline();
        if (deadline == TInstant::Max()) {
            return MAX_API_TIMEOUT;
        }

        auto timeout = deadline - TInstant::Now();

        return Min(timeout, MAX_API_TIMEOUT);
    }

    /// Call this function upon receiving `NActors::TEvents::TEvWakeup`.
    void SendTimeoutAndPassAway() {
        GetRequestCtx()->ReplyError(
            grpc::DEADLINE_EXCEEDED,
            TStringBuilder{} << "deadline exceeded after " << GetRequestTimeout());
        NActors::TActorBootstrapped<TDerived>::PassAway();
    }

    /// Get metrics registry used for GRPC server. Can never be nullptr.
    NMonitoring::TMetricRegistry* GetRegistry() const {
        return Registry_;
    }

private:
    void BeforeRegister(
            ::NGrpc::IRequestContextBase* ctx,
            NMonitoring::TMetricRegistry& registry,
            NPrivate::TActorSystemMetrics endpoint,
            NPrivate::TActorSystemMetrics total)
    {
        RequestCtx_ = ctx;

        Registry_ = &registry;

        Endpoint_ = endpoint;
        Endpoint_.OnCallRegistered();

        Total_ = total;
        Total_.OnCallRegistered();

        CreateTime_ = std::chrono::steady_clock::now();
    }

private:
    ::NGrpc::IRequestContextBase* RequestCtx_ = nullptr;
    NMonitoring::TMetricRegistry* Registry_ = nullptr;
    NPrivate::TActorSystemMetrics Endpoint_;
    NPrivate::TActorSystemMetrics Total_;
    std::chrono::steady_clock::time_point CreateTime_;
};

/// A macro to generate all boilerplate required by `THandler`.
///
///
/// # Usage
///
/// ```
/// SOLOMON_RPC_HANDLER(generated::GreetingService, Greet) {
///     // This is a normal class body -- implement as usual.
///
///     void Handle(const NActors::TActorContext& ctx) override {
///         // ...
///     }
/// };
/// ```
#define SOLOMON_RPC_HANDLER(service, method) \
    class T##method##Handler;                                                                                   \
    class T##method##HandlerBase: public ::NSolomon::NGrpc::THandler<T##method##Handler> {                      \
    public:                                                                                                     \
        using TGeneratedService = service;                                                                      \
        static constexpr TStringBuf Name = TStringBuf(#method);                                                \
        static constexpr auto Callback = &TGeneratedService::AsyncService::Request##method;                     \
        using TRequest =                                                                                        \
            typename ::NSolomon::NGrpc::NPrivate::TSignature<std::remove_const_t<decltype(Callback)>>::Request; \
        using TResponse =                                                                                       \
            typename ::NSolomon::NGrpc::NPrivate::TSignature<std::remove_const_t<decltype(Callback)>>::Response;\
    };                                                                                                          \
    class T##method##Handler final: public T##method##HandlerBase

/// RPC service that uses actors to handle requests (`THandler` above).
///
/// @tparam TGeneratedService
///     service class as generated by protoc. Must have a `static constexpr char const* service_full_name()`
///     function and an `AsyncService` nested class.
///
///
/// # Usage
///
/// Create a `THandler` subclass for each request. Then, derive from this class
/// and implement `InitHandlers` to register all handlers:
///
/// ```
/// class TMyService final: public NSolomon::NGrpc::TService<generated::GreetingService> {
/// public:
///     void InitHandlers(grpc::ServerCompletionQueue* cq) override {
///         Register<TGreetHandler>(cq);
///         // ...
///     }
/// }
/// ```
template <typename TGeneratedService>
class TService: public ::NGrpc::TGrpcServiceBase<TGeneratedService>, private TNonCopyable {
public:
    using TSelf = TService<TGeneratedService>;
    using TBase = ::NGrpc::TGrpcServiceBase<TGeneratedService>;

public:
    TService(NActors::TActorSystem& actorSystem, NMonitoring::TMetricRegistry& registry, ui32 executorPool)
        : ActorSystem_(actorSystem)
        , Registry_(registry)
        , ExecutorPool_(executorPool)
    {
    }

public:
    /// Implement this function. In implementation, call `RegisterHandler` for each procedure in the service.
    virtual void InitHandlers(grpc::ServerCompletionQueue* cq) = 0;

protected:
    /// Register a new request handler.
    ///
    /// For each request, an instance of `THandler` actor will be created and initialized with arguments `args`.
    template <typename THandler, typename... TArgs>
    void RegisterHandler(grpc::ServerCompletionQueue* cq, TArgs&&... args) {
        static_assert(
            std::is_base_of_v<NPrivate::THandlerTag, THandler>,
            "handler must be derived from 'NSolomon::NGrpc::THandler'");
        static_assert(
            std::is_same_v<TGeneratedService, typename THandler::TGeneratedService>,
            "trying to register a handler from a different service");

        using TRequest = typename THandler::TRequest;
        using TResponse = typename THandler::TResponse;

        NPrivate::TActorSystemMetrics endpoint{
            Registry_, TStringBuilder{} << "/" << TGeneratedService::service_full_name() << "/" << THandler::Name};
        NPrivate::TActorSystemMetrics total{
            Registry_, "total"};

        MakeIntrusive<::NGrpc::TGRpcRequest<TRequest, TResponse, TSelf>>(
            this,
            TBase::GetService(),
            cq,
            [=](::NGrpc::IRequestContextBase* ctx) {
                auto handler = MakeHolder<THandler>(args...);
                handler->BeforeRegister(ctx, Registry_, endpoint, total);
                ActorSystem_.Register(handler.Release(), NActors::TMailboxType::Simple, ExecutorPool_);
            },
            THandler::Callback,
            THandler::Name.data(),
            Logger_,
            ::NGrpc::FakeCounterBlock())->Run();
    }

    template <typename TGrpcMethod, typename... TArgs>
    void RegisterSingleHandler(grpc::ServerCompletionQueue* cq, TGrpcMethod grpcMethod, const char* name, NActors::IActor* handler) {
        using TRequest = typename NPrivate::TSignature<TGrpcMethod>::Request;
        using TResponse = typename NPrivate::TSignature<TGrpcMethod>::Response;

        auto handlerId = ActorSystem_.Register(handler, NActors::TMailboxType::HTSwap, ExecutorPool_);
        MakeIntrusive<::NGrpc::TGRpcRequest<TRequest, TResponse, TSelf>>(
                this,
                TBase::GetService(),
                cq,
                [this, handlerId](::NGrpc::IRequestContextBase* ctx) {
                    ActorSystem_.Send(handlerId, new NGrpc::TServerEvents::TRequest{ctx, TInstant::Now()});
                },
                grpcMethod,
                name,
                Logger_,
                ::NGrpc::FakeCounterBlock())->Run();
    }

public:
    void InitService(grpc::ServerCompletionQueue* cq, ::NGrpc::TLoggerPtr logger) final {
        Logger_ = logger;
        InitHandlers(cq);
        CheckHandlers();
    }

    void SetGlobalLimiterHandle(::NGrpc::TGlobalLimiter* limiter) final {
        Limiter_ = limiter;
    }

    bool IncRequest() {
        return Limiter_->Inc();
    }

    void DecRequest() {
        Limiter_->Dec();
    }

    NMonitoring::TMetricRegistry& GetMetricRegistry() {
        return Registry_;
    }

private:
    // We use reflection to check that all functions have exactly one handler assigned.
    void CheckHandlers() {
        // TODO!
        // auto pool = google::protobuf::DescriptorPool::generated_pool();
        // Y_VERIFY(pool);
        // auto service = pool->FindServiceByName(google::protobuf::string(TGeneratedService::service_full_name()));
        // Y_VERIFY(service);
    }

private:
    NActors::TActorSystem& ActorSystem_;
    NMonitoring::TMetricRegistry& Registry_;
    const ui32 ExecutorPool_;
    ::NGrpc::TLoggerPtr Logger_;
    ::NGrpc::TGlobalLimiter* Limiter_ = nullptr;
};

} // namespace NSolomon::NGrpc
