#pragma once

#include <solomon/libs/cpp/grpc/server/handler.h>
#include <solomon/libs/cpp/grpc/server/ut/protos/math_service.grpc.pb.h>

#include <library/cpp/grpc/server/grpc_request.h>
#include <library/cpp/grpc/server/grpc_server.h>

namespace NSolomon {

// simple sync handler
class TAddHandler: public NGrpc::TBaseHandler<TAddHandler> {
public:
    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* args = req->GetMessage<yandex::monitoring::test::Args>();
        yandex::monitoring::test::Result result;
        result.set_value(args->left() + args->right());
        req->SendReply(&result);
        return EMode::Sync;
    }
};

// simple sync handler
class TSubHandler: public NGrpc::TBaseHandler<TSubHandler> {
public:
    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* args = req->GetMessage<yandex::monitoring::test::Args>();
        yandex::monitoring::test::Result result;
        result.set_value(args->left() - args->right());
        req->SendReply(&result);
        return EMode::Sync;
    }
};

// offload actual calculations to a different actor
class TMulHandler: public NGrpc::TBaseHandler<TMulHandler>, TPrivateEvents {
    enum {
        Calc = SpaceBegin,
        CalcResult,
        End
    };
    static_assert(SpaceBegin < End, "too many events");

    struct TEvCalc: public NActors::TEventLocal<TEvCalc, Calc> {
        double Left;
        double Right;

        explicit TEvCalc(double left, double right) noexcept
            : Left{left}
            , Right{right}
        {
        }
    };

    struct TEvCalcResult: public NActors::TEventLocal<TEvCalcResult, CalcResult> {
        double Result;

        explicit TEvCalcResult(double result) noexcept
            : Result{result}
        {
        }
    };

    class TAsyncMultiplier: public NActors::TActor<TAsyncMultiplier> {
    public:
        TAsyncMultiplier()
            : NActors::TActor<TAsyncMultiplier>{&TThis::StateFunc}
        {
        }

        STATEFN(StateFunc) {
            switch (ev->GetTypeRewrite()) {
                hFunc(TEvCalc, OnCalc)
            }
        }

        void OnCalc(TEvCalc::TPtr& ev) {
            Send(ev->Sender, new TEvCalcResult{ev->Get()->Left * ev->Get()->Right}, 0, ev->Cookie);
        }
    };

public:
    STATEFN(HandleEvent) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvCalcResult, OnCalcResult)
        }
    }

    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* args = req->GetMessage<yandex::monitoring::test::Args>();
        auto handlerId = Register(new TAsyncMultiplier);
        Send(handlerId, new TEvCalc{args->left(), args->right()}, 0, req->ToCookie());
        return EMode::Async;
    }

    void OnCalcResult(TEvCalcResult::TPtr& ev) {
        auto req = NGrpc::TRequestState::FromCookie(ev->Cookie);
        yandex::monitoring::test::Result result;
        result.set_value(ev->Get()->Result);
        req->SendReply(&result);
    }
};

// delaying calculations
class TDivHandler: public NGrpc::TBaseHandler<TDivHandler> {
public:
    EMode HandleRequest(NGrpc::TRequestState* req) {
        auto* args = req->GetMessage<yandex::monitoring::test::Args>();
        auto result = std::make_unique<yandex::monitoring::test::Result>();
        result->set_value(args->left() / args->right());

        // non immediate response
        NActors::TActivationContext::Schedule(TDuration::MilliSeconds(300), new NActors::IEventHandle{
            SelfId(),
            {},
            new NGrpc::TServerEvents::TResponse{std::move(result)},
            0,
            req->ToCookie()
        });
        return EMode::Async;
    }
};

template <typename T>
struct TAsyncUnary;

template <typename TService, typename TRequest, typename TResponse>
struct TAsyncUnary<void (TService::*)(grpc::ServerContext*, TRequest*, grpc::ServerAsyncResponseWriter<TResponse>*, grpc::CompletionQueue*, grpc::ServerCompletionQueue*, void*)> {
    using Request = TRequest;
    using Response = TResponse;
};

class TMathServiceActor: public ::NGrpc::TGrpcServiceBase<yandex::monitoring::test::MathService> {
public:
    explicit TMathServiceActor(NActors::TActorSystem& actorSystem) noexcept
        : ActorSystem_{actorSystem}
    {
    }

    void InitService(grpc::ServerCompletionQueue* cq, ::NGrpc::TLoggerPtr logger) override {
        using yandex::monitoring::test::MathService;

        Cq_ = cq;
        Logger_ = std::move(logger);

        RegisterHandler<&MathService::AsyncService::RequestAdd, TAddHandler>("Add");
        RegisterHandler<&MathService::AsyncService::RequestSub, TSubHandler>("Sub");
        RegisterHandler<&MathService::AsyncService::RequestMul, TMulHandler>("Mul");
        RegisterHandler<&MathService::AsyncService::RequestDiv, TDivHandler>("Div");
    }

    bool IncRequest() {
        return true;
    }

    void DecRequest() {
    }

private:
    template <auto Method, typename THandler>
    void RegisterHandler(const char* name) {
        auto handlerId = ActorSystem_.Register(new THandler);

        using TReq = typename TAsyncUnary<decltype(Method)>::Request;
        using TResp = typename TAsyncUnary<decltype(Method)>::Response;

        MakeIntrusive<::NGrpc::TGRpcRequest<TReq, TResp, TMathServiceActor>>(
                this,
                &Service_,
                Cq_,
                [this, handlerId](::NGrpc::IRequestContextBase* ctx) {
                    ActorSystem_.Send(handlerId, new NGrpc::TServerEvents::TRequest{ctx, TInstant::Now()});
                },
                Method,
                name,
                Logger_,
                ::NGrpc::FakeCounterBlock())->Run();
    }

private:
    NActors::TActorSystem& ActorSystem_;
    grpc::ServerCompletionQueue* Cq_;
    ::NGrpc::TLoggerPtr Logger_;
};

} // namespace NSolomon
