#pragma once

#include "stats_table.h"

#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/libs/cpp/trace/trace_fwd.h>

#include <library/cpp/grpc/server/grpc_request_base.h>

#include <library/cpp/grpc/server/grpc_request.h>

namespace NSolomon::NGrpc {

class TMeasuredGrpcRequestCtx: public ::NGrpc::IRequestContextBase {
public:
    TMeasuredGrpcRequestCtx(::NGrpc::IRequestContextBase* original, std::shared_ptr<TStatsTable> table, TDuration minInterestingTime, TInstant timestamp)
        : Delegate_{original}
        , StartedAt_{timestamp}
        , Table_{std::move(table)}
        , MinimumInterestingTime_{minInterestingTime}
    {
        Table_->RecordStart(StartedAt_, Delegate_);
    }

    // all member functions will look similarly, for instance

    const NProtoBuf::Message* GetRequest() const override {
        return Delegate_->GetRequest();
    }

    ::NGrpc::TAuthState& GetAuthState() override {
        return Delegate_->GetAuthState();
    }

    TInstant Deadline() const override {
        return Delegate_->Deadline();
    }

    TSet<TStringBuf> GetPeerMetaKeys() const override {
        return Delegate_->GetPeerMetaKeys();
    }

    TVector<TStringBuf> GetPeerMetaValues(TStringBuf key) const override {
        return Delegate_->GetPeerMetaValues(key);
    }

    grpc_compression_level GetCompressionLevel() const override {
        return Delegate_->GetCompressionLevel();
    }

    google::protobuf::Arena* GetArena() override {
        return Delegate_->GetArena();
    }

    void AddTrailingMetadata(const TString& key, const TString& value) override {
        Delegate_->AddTrailingMetadata(key, value);
    }

    void UseDatabase(const TString& database) override {
        Delegate_->UseDatabase(database);
    }

    void SetNextReplyCallback(TOnNextReply&& cb) override {
        Delegate_->SetNextReplyCallback(std::move(cb));
    }

    void FinishStreamingOk() override {
        Delegate_->FinishStreamingOk();
    }

    TAsyncFinishResult GetFinishFuture() override {
        return Delegate_->GetFinishFuture();
    }

    TString GetPeer() const override {
        return Delegate_->GetPeer();
    }

    bool SslServer() const override {
        return Delegate_->SslServer();
    }

    void SetTraceId(NTracing::TTraceId traceId) {
        TraceId_ = std::move(traceId);
    }

    // except Reply* functions

    void Reply(NProtoBuf::Message* resp, ui32 status) override {
        std::unique_ptr<IRequestContextBase> deferDelete(this); // hack to delete current object after this function ends
        auto elapsed = TInstant::Now() - StartedAt_;
        if (elapsed >= MinimumInterestingTime_ && elapsed >= Table_->GetMinimumInterestingTime()) {
            TRow row{Delegate_, StartedAt_, elapsed};
            TAdditionalInfo info{Delegate_->GetRequest()->DebugString(), resp->DebugString(), TraceId_};
            Delegate_->Reply(resp, status);
            Table_->RecordFinish(std::move(row), std::move(info), Delegate_);
        } else {
            Delegate_->Reply(resp, status);
            Table_->RemoveFromUnfinished(Delegate_, TraceId_);
        }
    }

    void Reply(grpc::ByteBuffer* resp, ui32 status = 0) override { // No response DebugString
        std::unique_ptr<IRequestContextBase> deferDelete(this);
        auto elapsed = TInstant::Now() - StartedAt_;
        if (elapsed >= MinimumInterestingTime_ && elapsed >= Table_->GetMinimumInterestingTime()) {
            TRow row{Delegate_, StartedAt_, elapsed};
            TAdditionalInfo info{Delegate_->GetRequest()->DebugString(), "No Debug String, replied by ByteBuffer", TraceId_};
            Delegate_->Reply(resp, status);
            Table_->RecordFinish(std::move(row), std::move(info), Delegate_);
        } else {
            Delegate_->Reply(resp, status);
            Table_->RemoveFromUnfinished(Delegate_, TraceId_);
        }
    }

    void ReplyUnauthenticated(const TString& in) override { // No response DebugString
        std::unique_ptr<IRequestContextBase> deferDelete(this);
        auto elapsed = TInstant::Now() - StartedAt_;
        if (elapsed >= MinimumInterestingTime_ && elapsed >= Table_->GetMinimumInterestingTime()) {
            TRow row{Delegate_, StartedAt_, elapsed};
            TAdditionalInfo info{Delegate_->GetRequest()->DebugString(), "No Debug String, replied by ByteBuffer", TraceId_};
            Delegate_->ReplyUnauthenticated(in);
            Table_->RecordFinish(std::move(row), std::move(info), Delegate_);
        } else {
            Delegate_->ReplyUnauthenticated(in);
            Table_->RemoveFromUnfinished(Delegate_, TraceId_);
        }
    }

    void ReplyError(grpc::StatusCode code, const TString& msg, const TString& details = "") override {
        std::unique_ptr<IRequestContextBase> deferDelete(this);
        auto elapsed = TInstant::Now() - StartedAt_;
        if (elapsed >= MinimumInterestingTime_ && elapsed >= Table_->GetMinimumInterestingTime()) {
            TRow row{Delegate_, StartedAt_, elapsed};
            TAdditionalInfo info{Delegate_->GetRequest()->DebugString(), ToString(code) + ", Message: " + msg, TraceId_};
            Delegate_->ReplyError(code, msg, details);
            Table_->RecordFinish(std::move(row), std::move(info), Delegate_);
        } else {
            Delegate_->ReplyError(code, msg, details);
            Table_->RemoveFromUnfinished(Delegate_, TraceId_);
        }
    }

private:
    IRequestContextBase* Delegate_;
    TInstant StartedAt_;
    std::shared_ptr<TStatsTable> Table_;
    TDuration MinimumInterestingTime_;
    NTracing::TTraceId TraceId_{};
};

inline NTracing::TSpanId MakeTraceContextFromGrpc(::NGrpc::IRequestContextBase* grpcReqCtx) {
    auto* ctx = dynamic_cast<NGrpc::TMeasuredGrpcRequestCtx*>(grpcReqCtx);
    if (ctx == nullptr) {
        return {};
    }

    // TODO
    ui8 verbosity = 15; // max
    ui32 ttl = 4095; // max
    auto span = NTracing::TSpanId::NewTraceId(verbosity, ttl);
    ctx->SetTraceId(NTracing::TTraceId::From(span));
    return span;
}

} // namespace NSolomon::NGrpc
