#include "stats_page.h"
#include "stats_filter.h"
#include "stats_table.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/generic/algorithm.h>
#include <util/stream/format.h>

using namespace NSolomon::NSelfMon;
using namespace yandex::monitoring::selfmon;

namespace NSolomon::NGrpc {
namespace {

const TString PagePath = "/grpc_server";

template <typename T>
std::optional<T> ParseParam(const TQuickCgiParam& params, TStringBuf name) {
    if (TStringBuf valueStr = params.Get(name)) {
        T value{};
        if (TryFromString(valueStr, value)) {
            return value;
        }
    }
    return std::nullopt;
}

struct TStatsParams {
public:
    TString From;
    TString Type;
    ERequestStatus Status = ERequestStatus::Finished;
    TString ActiveTab;
    i32 RequestsLimit{20};

    friend IOutputStream& operator<<(IOutputStream& out, const TStatsParams& params) {
        out << "tab=" << params.ActiveTab << "&limit=" <<
            params.RequestsLimit << "&from=" << params.From <<
            "&type=" << params.Type << "&status=" << params.Status;
        return out;
    }
};

void RenderTabs(Tabs* tabs, TStatsParams params) {
    if (auto* tab = tabs->add_tabs()) {
        tab->set_active(params.ActiveTab.empty() || params.ActiveTab == "requests");
        if (auto* ref = tab->mutable_reference()) {
            ref->set_title("Requests");
            ref->set_page(PagePath);
            TString curTab = std::move(params.ActiveTab);
            params.ActiveTab = "requests";
            ref->set_args(TStringBuilder{} << params);
            params.ActiveTab = std::move(curTab);
        }
    }
    if (auto* tab = tabs->add_tabs()) {
        tab->set_active(params.ActiveTab == "settings");
        if (auto* ref = tab->mutable_reference()) {
            ref->set_title("Settings");
            ref->set_page(PagePath);
            TString curTab = std::move(params.ActiveTab);
            params.ActiveTab = "settings";
            ref->set_args(TStringBuilder{} << params);
        }
    }
}

void RenderComponentsTable(Table* table, const TStatsParams& params, TVector<TRow> data) {
    table->set_numbered(true);

    auto* fromCol = table->add_columns();
    fromCol->set_title("From");
    auto* fromValues = fromCol->mutable_string();

    auto* typeCol = table->add_columns();
    typeCol->set_title("Request Type");
    auto* typeValues = typeCol->mutable_string();

    auto* startedCol = table->add_columns();
    startedCol->set_title("Started Time");
    auto* startedValues = startedCol->mutable_time();

    auto* deadlineCol = table->add_columns();
    deadlineCol->set_title("Deadline");
    auto* deadlineValues = deadlineCol->mutable_string();

    auto* finishedCol = table->add_columns();
    finishedCol->set_title("Finished");
    auto* finishedValues= finishedCol->mutable_string();

    auto* elapsedCol = table->add_columns();
    elapsedCol->set_title("Elapsed Time");
    auto* elapsedValues = elapsedCol->mutable_duration();

    auto* detailsCol = table->add_columns();
    detailsCol->set_title("Details");
    auto* detailsValues = detailsCol->mutable_reference();

    for (size_t i = 0; i < data.size(); ++i) {
        const auto& r = data[i];

        fromValues->add_values(r.From);
        typeValues->add_values(r.Type);
        startedValues->add_values(r.StartedAt.GetValue());
        deadlineValues->add_values(r.Deadline == TInstant::Max() ? "Not Defined" : r.Deadline.ToStringUpToSeconds());
        finishedValues->add_values(r.Finished ? "Yes" : "No");
        elapsedValues->add_values(r.Finished ? r.ElapsedTime.GetValue(): (TInstant::Now() - r.StartedAt).GetValue());

        auto* ref = detailsValues->add_values();
        ref->set_title("Details");
        ref->set_args(TStringBuilder{} << params << "&id=" << r.Id);
        ref->set_page(PagePath);
    }
}

NTracing::TTraceId RenderDetailAndGetTraceId(Page* page, ui64 id, const TStatsParams& params, std::optional<std::pair<TRow, TAdditionalInfo>> optInfo) {
    page->set_title("Details for request with id " + ToString(id));

    auto* grid = page->mutable_grid();
    if (optInfo == std::nullopt) {
        if (auto* row = grid->add_rows()) {
            auto* h = row->add_columns()->mutable_component()->mutable_heading();
            h->set_level(4);
            h->set_content("Error, while getting Additional info");
        }
        if (auto* row = grid->add_rows()) {
            auto* ref = row->add_columns()->mutable_component()->mutable_value()->mutable_reference();
            ref->set_title("Back");
            ref->set_page(PagePath);
            ref->set_args(TStringBuilder{} << params);
        }
        return {};
    }
    auto info = optInfo.value();
    if (auto* row = grid->add_rows()) {
        if (auto* col = row->add_columns()) {
            auto* object = col->mutable_component()->mutable_object();
            auto AddField = [&](TString name, TString value) {
                auto* field = object->add_fields();
                field->set_name(std::move(name));
                auto* fieldValue = field->mutable_value();
                fieldValue->set_string(std::move(value));
            };

            AddField("From", info.first.From);
            AddField("Request Type", info.first.Type);
            {
                auto* field = object->add_fields();
                field->set_name("Started Time");
                auto* fieldValue = field->mutable_value();
                fieldValue->set_time(info.first.StartedAt.GetValue());
            }
            if (info.first.Deadline == TInstant::Max()) {
                AddField("Deadline", "Not Defined");
            } else {
                auto* field = object->add_fields();
                field->set_name("Deadline");
                auto* fieldValue = field->mutable_value();
                fieldValue->set_time(info.first.Deadline.GetValue());
            }
            AddField("Finished", info.first.Finished ? "Yes" : "No");
            {
                auto* field = object->add_fields();
                field->set_name("Elapsed Time");
                auto* fieldValue = field->mutable_value();
                fieldValue->set_duration(info.first.ElapsedTime.GetValue());
            }
        }
    }

    if (auto* row = grid->add_rows()) {
        auto* h = row->add_columns()->mutable_component()->mutable_heading();
        h->set_level(4);
        h->set_content("Request Debug String");
    }
    if (auto* row = grid->add_rows()) {
        auto* code = row->add_columns()->mutable_component()->mutable_code();
        code->set_content(info.second.RequestDebugString);
    }
    if (auto* row = grid->add_rows()) {
        auto* h = row->add_columns()->mutable_component()->mutable_heading();
        h->set_level(4);
        h->set_content("Response Debug String");
    }
    if (auto* row = grid->add_rows()) {
        auto* code = row->add_columns()->mutable_component()->mutable_code();
        code->set_content(info.second.ResponseDebugString);
    }
    if (auto* row = grid->add_rows()) {
        auto* ref = row->add_columns()->mutable_component()->mutable_value()->mutable_reference();
        ref->set_title("Back");
        ref->set_page(PagePath);
        ref->set_args(TStringBuilder{} << params);
    }

    return info.second.TraceId;
}

void RenderTraceSpans(Page* page, NTracing::TTraceId traceId, TVector<NTracing::TSpan>& spans) {
    auto* grid = page->mutable_grid();

    int backRow = grid->mutable_rows()->size() - 1;

    if (auto* object = grid->add_rows()->add_columns()->mutable_component()->mutable_object()) {
        auto AddField = [&](TString name, TString value) {
            auto* field = object->add_fields();
            field->set_name(std::move(name));
            auto* fieldValue = field->mutable_value();
            fieldValue->set_string(std::move(value));
        };

        TString traceIdStr = traceId.Hex();
        traceIdStr.to_lower();
        AddField("TraceId", traceIdStr);
    }

    TInstant min = TInstant::Max();
    for (const auto& span: spans) {
        if (span.Begin < min) {
            min = span.Begin;
        }
        if (span.End && (span.End < min)) {
            min = span.End;
        }
    }
    
    if (auto* table = grid->add_rows()->add_columns()->mutable_component()->mutable_table()) {
        auto addColumn = [table] (const TString& name) {
            auto* col = table->add_columns();
            col->set_title(name);
            return col;
        };

        auto* spanId = addColumn("Span")->mutable_string();
        auto* parentId = addColumn("Parent")->mutable_string();
        auto* descr = addColumn("Description")->mutable_string();
        auto* from = addColumn("From")->mutable_duration();
        auto* to = addColumn("To")->mutable_duration();
        auto* duration = addColumn("Duration")->mutable_duration();

        for (auto& span: spans) {
            descr->add_values(span.Description);
            spanId->add_values(TStringBuilder{} << Hex(span.Id, HF_FULL) << (span.End ? ' ' : '*'));
            parentId->add_values(TStringBuilder{} << Hex(span.ParentId, HF_FULL));
            from->add_values((span.Begin - min).GetValue());
            auto end = span.End ? span.End : NSolomon::NTracing::TraceingClockNow();
            to->add_values((end - min).GetValue());
            duration->add_values((end - span.Begin).GetValue());
        }
    }

    /* js TracePlot is exteeeemly slow, so use text representation atm
    if (auto* traceplot = grid->add_rows()->add_columns()->mutable_component()->mutable_trace_plot()) {
        for (auto& span: spans) {
            auto* pbSpan = traceplot->add_spans();
            pbSpan->set_id(TStringBuilder{} << Hex(span.Id, HF_FULL));
            pbSpan->set_parent(TStringBuilder{} << Hex(span.ParentId, HF_FULL));
            pbSpan->set_description(span.Description);
            pbSpan->set_begin((span.Begin - min).GetValue());
            if (span.End == TInstant::Zero()) {
                pbSpan->set_finished(false);
                pbSpan->set_end(0);
            } else {
                pbSpan->set_finished(true);
                pbSpan->set_end((span.End - min).GetValue());
            }
        }
    }
    */

    int traceRow = grid->mutable_rows()->size();
    if (auto* row = grid->add_rows()) {
        auto* h = row->add_columns()->mutable_component()->mutable_heading();
        h->set_level(4);
        h->set_content("Trace");
    }

    grid->mutable_rows()->SwapElements(backRow, traceRow);
}

void RenderErrorDetail(Page* page, TStringBuf id, const TStatsParams& params) {
    page->set_title("I can not parse the number from the string \"" + ToString(id) + "\"");

    auto* grid = page->mutable_grid();

    if (auto* row = grid->add_rows()) {
        auto* ref = row->add_columns()->mutable_component()->mutable_value()->mutable_reference();
        ref->set_title("Back");
        ref->set_page(PagePath);
        ref->set_args(TStringBuilder{} << params);
    }
}

void RenderFilterGrid(Component* component, const TStatsParams& params) {
    auto* form = component->mutable_form();
    form->set_layout(FormLayout::Horizontal);
    form->set_method(yandex::monitoring::selfmon::FormMethod::Get);
    {
        auto* item = form->add_items();
        auto* input = item->mutable_input();
        input->set_type(yandex::monitoring::selfmon::InputType::Text);
        input->set_placeholder("From");
        input->set_name("from");
        input->set_value(params.From);
    }
    {
        auto* item = form->add_items();
        auto* input = item->mutable_input();
        input->set_type(yandex::monitoring::selfmon::InputType::Text);
        input->set_placeholder("Type");
        input->set_name("type");
        input->set_value(params.Type);
    }
    {
        auto* item = form->add_items();
        auto* select = item->mutable_select();
        select->set_name("status");
        {
            auto* opt = select->Addoptions();
            opt->set_value("finished");
            opt->set_title("Finished");
            opt->set_selected(params.Status == ERequestStatus::Finished);
        }
        {
            auto* opt = select->Addoptions();
            opt->set_value("in_flight");
            opt->set_title("In Flight");
            opt->set_selected(params.Status == ERequestStatus::InFlight);
        }
        {
            auto* opt = select->Addoptions();
            opt->set_value("any");
            opt->set_title("Any");
            opt->set_selected(params.Status == ERequestStatus::Any);
        }
    }

    auto* apply = form->add_submit();
    apply->set_title("Apply");
}

void RenderSettingsTab(Grid* grid, ui32 availablePeriod, ui32 maximumSize, ui32 minElapsed, ui32 samplingRate, TString captureType) {
    if (auto* r = grid->add_rows()) {
        if (auto* col = r->add_columns()) {
            col->set_width(4);
            auto* f = col->mutable_component()->mutable_form();
            f->set_layout(FormLayout::Vertical);
            f->set_method(FormMethod::Post);

            if (auto* item = f->add_items()) {
                item->set_label("Minimum request elapsed time");
                item->set_help("In milliseconds");
                auto* input = item->mutable_input();
                input->set_type(InputType::IntNumber);
                input->set_name("minElapsed");
                input->set_value(ToString(minElapsed));
            }
            if (auto* item = f->add_items()) {
                item->set_label("Available Period");
                item->set_help("The time that the request is stored in the table, in seconds");
                auto* input = item->mutable_input();
                input->set_type(InputType::IntNumber);
                input->set_name("availablePeriod");
                input->set_value(ToString(availablePeriod));
            }
            if (auto* item = f->add_items()) {
                item->set_label("Count of Interesting Requests");
                auto* input = item->mutable_input();
                input->set_type(InputType::IntNumber);
                input->set_name("maximumSize");
                input->set_value(ToString(maximumSize));
            }
            if (auto* item = f->add_items()) {
                item->set_label("Sampling Rate");
                auto* input = item->mutable_input();
                input->set_type(InputType::IntNumber);
                input->set_name("samplingRate");
                input->set_value(ToString(samplingRate));
            }
            if (auto* item = f->add_items()) {
                item->set_label("Type");
                item->set_help("Filter of types of captured requests"
                               "supports glob syntax");
                auto* input = item->mutable_input();
                input->set_placeholder("e.g *Read*");
                input->set_type(InputType::Text);
                input->set_name("captureType");

                input->set_value(std::move(captureType));
            }

            auto* submit = f->add_submit();
            submit->set_title("Save");
        }
    }
}

bool IsMatched(TStringBuf str, TStringBuf substr) noexcept {
    return !substr || str.Contains(substr) || IsGlobMatch(substr, str);
}

TVector<TRow> GetFinishedData(const std::shared_ptr<TStatsTable>& table, TStringBuf fromFilter, TStringBuf typeFilter) {
    TVector<TRow> data = table->GetFinishedData();
    if (!fromFilter && !typeFilter) {
        return data;
    }
    EraseIf(data, [&](const TRow& row) {
        return !IsMatched(row.From, fromFilter) || !IsMatched(row.Type, typeFilter);
    });
    return data;
}

TVector<TRow> GetUnfinishedData(const std::shared_ptr<TStatsTable>& table, TStringBuf fromFilter, TStringBuf typeFilter) {
    TVector<TRow> data = table->GetUnfinishedData();
    if (!fromFilter && !typeFilter) {
        return data;
    }
    EraseIf(data, [&](const TRow& row) {
        return !IsMatched(row.From, fromFilter) || !IsMatched(row.Type, typeFilter);
    });
    return data;
}

TVector<TRow> GetFilteredData(const std::shared_ptr<TStatsTable>& table, ERequestStatus status, TStringBuf fromFilter, TStringBuf typeFilter, int cnt) {
    TVector<TRow> data;
    if (status == ERequestStatus::Finished) {
        data = GetFinishedData(table, fromFilter, typeFilter);
    } else if (status == ERequestStatus::InFlight) {
        data = GetUnfinishedData(table, fromFilter, typeFilter);
    } else {
        data = GetFinishedData(table, fromFilter, typeFilter);
        TVector<TRow> unfinishedData = GetUnfinishedData(table, fromFilter, typeFilter);
        data.reserve(data.size() + unfinishedData.size());
        std::move(unfinishedData.begin(), unfinishedData.end(), std::back_inserter(data));
    }

    if (cnt == -1 || data.size() <= static_cast<ui32>(cnt)) {
        return data;
    }

    std::nth_element(data.begin(), data.begin() + cnt, data.end(), [](const TRow& a, const TRow& b){
        return a.ElapsedTime > b.ElapsedTime;
    });
    data.resize(cnt);
    return data;
}

void RenderMainPage(Page* page, TStatsParams params, const std::shared_ptr<TStatsTable>& table,
        const TStatsTableSettings& settings, const TStatsFilterConf& conf) {
    auto* grid = page->mutable_grid();
    if (auto* r = grid->add_rows()) {
        auto* component = r->add_columns()->mutable_component();
        RenderTabs(component->mutable_tabs(), params);
    }
    if (!params.ActiveTab || params.ActiveTab == "requests") {
        page->set_title("Slow gRPC Requests");

        auto* row = grid->add_rows();
        auto* component = row->add_columns()->mutable_component();
        RenderFilterGrid(component, params);

        if (auto* row = grid->add_rows()) {
            if (auto* col = row->add_columns()) {
                RenderComponentsTable(col->mutable_component()->mutable_table(), params,
                        GetFilteredData(table, params.Status, params.From, params.Type, params.RequestsLimit));
            }
        }

        if (auto* r2 = grid->add_rows()) {
            auto* ref = r2->add_columns()->mutable_component()->mutable_value()->mutable_reference();
            if (params.RequestsLimit != -1) {
                ref->set_title("Show all");
                ref->set_page(PagePath);
                params.RequestsLimit = -1;
                ref->set_args(TStringBuilder{} << params);
            } else {
                ref->set_title("Show first 20");
                ref->set_page(PagePath);
                params.RequestsLimit = 20;
                ref->set_args(TStringBuilder{} << params);
            }
        }
    } else {
        page->set_title("Settings");
        RenderSettingsTab(grid, settings.AvailablePeriod.Seconds(), settings.MaximumSize,
                conf.MinimumInterestingTime.MilliSeconds(), conf.SamplingRate, conf.CaptureType);
    }
}

class TStatsPage: public NActors::TActor<TStatsPage> {
public:
    TStatsPage(std::shared_ptr<TStatsTable> table, std::shared_ptr<IStatsFilter> filter)
        : NActors::TActor<TStatsPage>(&TThis::StateFunc)
        , Filter_{std::move(filter)}
        , Table_{std::move(table)}
    {
        Y_ENSURE(Filter_, "StatsFilter must be initialized");
        Y_ENSURE(Table_, "StatsTable must be initialized");
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TEvPageDataReq, OnRequest)
            hFunc(NTracing::TTracingEvents::TEvGetTraceResponse, OnTraceReceived)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(const TEvPageDataReq::TPtr& ev) {
        if (ev->Get()->HttpReq->Method == "POST") {
            TQuickCgiParam params{ev->Get()->HttpReq->Body};
            ProcessParamsActions(params);
        }

        TStatsParams params;

        TryFromString(ev->Get()->Param("limit"), params.RequestsLimit);
        params.From = ev->Get()->Param("from");
        params.Type = ev->Get()->Param("type");
        if (auto status = ev->Get()->Param("status")) {
            TryFromString(status, params.Status);
        } else {
            params.Status = ERequestStatus::Finished;
        }
        params.ActiveTab = ev->Get()->Param("tab");

        auto page = std::make_unique<Page>();
        NTracing::TTraceId traceId = {};
        if (auto ids = ev->Get()->Param("id")) {
            ui64 id = 0;
            if (TryFromString(ids, id)) {
                traceId = RenderDetailAndGetTraceId(page.get(), id, params, Table_->GetAdditionalInfo(id));
            } else {
                RenderErrorDetail(page.get(), ids, params);
            }
        } else {
            RenderMainPage(page.get(), params, Table_, Table_->GetTableSettings(), Filter_->GetFilterConfig());
        }
        if (traceId.Empty()) {
            Send(ev->Sender, new TEvPageDataResp{std::move(*page)});
            return;
        }

        auto request = std::make_unique<TRequest>(ev->Sender, std::move(page));

        Send(NTracing::TracingServiceId(), new NTracing::TTracingEvents::TEvGetTraceRequest{traceId},
                0, reinterpret_cast<ui64>(request.release()));
        
    }

    void OnTraceReceived(NTracing::TTracingEvents::TEvGetTraceResponse::TPtr& evPtr) {
        auto& ev = *evPtr->Get();
        auto request = std::unique_ptr<TRequest>(reinterpret_cast<TRequest*>(evPtr->Cookie));
        auto traceId = ev.TraceId;
        auto page = std::move(request->IncompletePage);

        RenderTraceSpans(page.get(), traceId, ev.Spans);

        Send(request->ReplyTo, new TEvPageDataResp{std::move(*page)});
    }

    void ProcessParamsActions(const TQuickCgiParam& params) {
        TStatsFilterConf conf;
        if (auto minElapsed = ParseParam<ui64>(params, "minElapsed")) {
            conf.MinimumInterestingTime = TDuration::MilliSeconds(minElapsed.value());
        }
        if (auto samplingRate = ParseParam<i64>(params, "samplingRate")) {
            conf.SamplingRate = samplingRate.value();
        }
        if (auto captureType = ParseParam<TString>(params, "captureType")) {
            conf.CaptureType = captureType.value();
        } else {
            conf.CaptureType = {};
        }
        Filter_->SetFilterConfig(conf);

        TStatsTableSettings settings;
        if (auto availablePeriod = ParseParam<i64>(params, "availablePeriod")) {
            settings.AvailablePeriod = TDuration::Seconds(availablePeriod.value());
        }
        if (auto maximumSize = ParseParam<i64>(params, "maximumSize")) {
            settings.MaximumSize = maximumSize.value();
        }
        Table_->SetTableSettings(settings);
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr& ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

private:
    std::shared_ptr<IStatsFilter> Filter_;
    std::shared_ptr<TStatsTable> Table_;

    struct TRequest {
        NActors::TActorId ReplyTo;
        std::unique_ptr<Page> IncompletePage;

        TRequest(NActors::TActorId replyTo, std::unique_ptr<Page> page)
            : ReplyTo(replyTo)
            , IncompletePage(std::move(page))
        {
        }
    };
};

} // namespace

std::unique_ptr<NActors::IActor> StatsPage(std::shared_ptr<TStatsTable> table, std::shared_ptr<IStatsFilter> filter) {
    return std::make_unique<TStatsPage>(std::move(table), std::move(filter));
}

} // namespace NSolomon::NGrpc
