#pragma once

#include "stats_table_metrics.h"
#include "stats_page.h"

#include <solomon/libs/cpp/glob/glob.h>
#include <solomon/libs/cpp/sync/rw_lock.h>
#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <library/cpp/grpc/server/grpc_request.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

#include <util/datetime/base.h>
#include <util/generic/algorithm.h>
#include <util/generic/hash.h>
#include <util/generic/queue.h>
#include <util/random/random.h>

#include <algorithm>
#include <functional>

namespace NSolomon::NGrpc {

struct TStatsTableSettings {
    TDuration AvailablePeriod{TDuration::Days(1)};
    size_t MaximumSize{50};
};

struct TStatsTableConf {
    NMonitoring::TMetricRegistry& Registry;
    TStatsTableSettings Settings;
};

struct TAdditionalInfo {
    TAdditionalInfo() = default;

    TAdditionalInfo(TString requestDebugString, TString responseDebugString, NTracing::TTraceId traceId = {})
        : RequestDebugString{std::move(requestDebugString)}
        , ResponseDebugString{std::move(responseDebugString)}
        , TraceId{std::move(traceId)}
    {
    }

    explicit TAdditionalInfo(TString requestDebugString)
        : RequestDebugString{std::move(requestDebugString)}
    {
    }

    TString RequestDebugString;
    TString ResponseDebugString;
    NTracing::TTraceId TraceId;
};

struct TRow {
    TRow()
        : From{}
        , Id{}
        , Type{}
        , StartedAt{TInstant::Now()}
        , Deadline{TInstant::Now()}
        , ElapsedTime{}
    {
    }

    TRow(TString from, TString type, const TInstant& startedAt, const TInstant& deadline, const TDuration elapsed)
        : From{std::move(from)}
        , Id{0}
        , Type{std::move(type)}
        , StartedAt{startedAt}
        , Deadline{deadline}
        , Finished{true}
        , ElapsedTime{elapsed}
    {
    }

    TRow(::NGrpc::IRequestContextBase* ctx, TInstant startedAt, TDuration elapsed)
        : From{ctx->GetPeer()}
        , Id{0}
        , Type{ctx->GetRequest()->GetTypeName()}
        , StartedAt{startedAt}
        , Deadline{ctx->Deadline()}
        , Finished{true}
        , ElapsedTime{elapsed}
    {
    }

    TRow(::NGrpc::IRequestContextBase* ctx, TInstant startedAt)
        : From{ctx->GetPeer()}
        , Id{reinterpret_cast<ui64>(ctx)}
        , Type{ctx->GetRequest()->GetTypeName()}
        , StartedAt{startedAt}
        , Deadline{ctx->Deadline()}
        , Finished{false}
    {
    }

    TString From;
    ui64 Id;
    TString Type;
    TInstant StartedAt;
    TInstant Deadline;
    bool Finished = false;
    TDuration ElapsedTime;
};

class TStatsTablePriorityQueue {
public:
    explicit TStatsTablePriorityQueue (TMetricRegistry& registry, const TDuration& availablePeriod, size_t maximumSize)
        : Metrics_(registry)
        , AvailablePeriod_{availablePeriod}
        , MaximumSize_{maximumSize}
    {
    }

    size_t Size() const {
        return Data_.size();
    }

    ui64 Insert(TRow row, TAdditionalInfo info) { // return current minimum time in the table
        TInstant cur = TInstant::Now();
        if (!MaximumSize_ || cur - row.StartedAt > AvailablePeriod_) {
            if (Data_.size() >= MaximumSize_) {
                return Data_[SortedElements_.top().GetId()].ElapsedTime.GetValue();
            } else {
                return 0;
            }
        }
        if (Data_.size() >= MaximumSize_) {
            size_t id = SortedElements_.top().GetId();
            if (Data_[id].ElapsedTime >= row.ElapsedTime && cur - Data_[id].StartedAt <= AvailablePeriod_) {
                return Data_[id].ElapsedTime.GetValue();
            }
            AdditionalInfo_.erase(Data_[id].Id);
            SortedElements_.pop();
            row.Id = IdSeq_++;
            AdditionalInfo_[row.Id] = std::make_pair(row, std::move(info));
            Data_[id] = std::move(row);
            SortedElements_.push({id, &Data_, &AvailablePeriod_});
        } else {
            row.Id = IdSeq_++;
            AdditionalInfo_[row.Id] = std::make_pair(row, std::move(info));
            Data_.push_back(row);
            SortedElements_.push({Data_.size() - 1u, &Data_, &AvailablePeriod_});
        }
        Metrics_.WriteRows(AdditionalInfo_.size());
        if (Data_.size() >= MaximumSize_) {
            return Data_[SortedElements_.top().GetId()].ElapsedTime.GetValue();
        } else {
            return 0;
        }
    }

    TVector<TRow> GetData() const {
        return Data_;
    }

    std::pair<TRow, TAdditionalInfo> GetAdditionalInfo(ui32 id) const {
        return AdditionalInfo_.at(id);
    }

    bool Contains(ui32 id) const {
        return AdditionalInfo_.contains(id);
    }

    class TElement {
    public:
        bool operator < (const TElement& other) const {
            if (TInstant::Now() - (*Data_)[Id_].StartedAt > (*AvailablePeriod_)) {
                return false;
            }
            if (TInstant::Now() - (*Data_)[other.Id_].StartedAt > (*AvailablePeriod_)) {
                return true;
            }
            return (*Data_)[Id_].ElapsedTime > (*Data_)[other.Id_].ElapsedTime;
        }

        TElement& operator = (const TElement& other) = default;

        TElement(const TElement& other) = default;

        TElement(size_t other, TVector<TRow>* data, const TDuration* AvailablePeriod)
            : Id_{other}
            , Data_{data}
            , AvailablePeriod_{AvailablePeriod}
        {
        }

        size_t GetId() const {
            return Id_;
        }

    private:
        size_t Id_;
        TVector<TRow>* Data_;
        const TDuration* AvailablePeriod_;
    };

    ui32 GetMaximumSize() const {
        return MaximumSize_;
    }

    ui64 SetMaximumSize(ui32 value) { // return current minimum time in the table
        MaximumSize_ = value;
        if (MaximumSize_ > SortedElements_.size()) {
            return 0;
        } else if (MaximumSize_ == SortedElements_.size()) {
            if (Data_.size() >= MaximumSize_) {
                return Data_[SortedElements_.top().GetId()].ElapsedTime.GetValue();
            } else {
                return 0;
            }
        }
        TVector<TRow> newData;
        TVector<TAdditionalInfo> info;
        newData.reserve(MaximumSize_);
        info.reserve(MaximumSize_);
        while (SortedElements_.size() > MaximumSize_) {
            size_t id = SortedElements_.top().GetId();
            AdditionalInfo_.erase(Data_[id].Id);
            SortedElements_.pop();
        }
        while (!SortedElements_.empty()) {
            size_t id = SortedElements_.top().GetId();
            info.push_back(AdditionalInfo_[Data_[id].Id].second);
            AdditionalInfo_.erase(Data_[id].Id);
            SortedElements_.pop();
            newData.push_back(Data_[id]);
        }
        std::swap(Data_, newData);
        for (size_t i = 0; i < Data_.size(); ++i) {
            AdditionalInfo_[Data_[i].Id] = std::make_pair(Data_[i], std::move(info[i]));
            SortedElements_.push({i, &Data_, &AvailablePeriod_});
        }
        Metrics_.WriteRows(AdditionalInfo_.size());
        if (Data_.size() >= MaximumSize_) {
            return Data_[SortedElements_.top().GetId()].ElapsedTime.GetValue();
        } else {
            return 0;
        }
    }

    TDuration GetAvailablePeriod() const {
        return AvailablePeriod_;
    }

    void SetAvailablePeriod(const TDuration& value) {
        AvailablePeriod_ = value;
    }

    TStatsTableSettings GetTableSettings() const {
        return TStatsTableSettings{AvailablePeriod_, MaximumSize_};
    }

private:
    TVector<TRow> Data_;
    THashMap<ui32, std::pair<TRow, TAdditionalInfo>> AdditionalInfo_;
    TPriorityQueue<TElement> SortedElements_;
    TStatsTableMetrics Metrics_;
    TDuration AvailablePeriod_;
    size_t MaximumSize_;
    ui64 IdSeq_ = 1;
};

class TStatsTable {
public:
    using TUnfinishedTable = absl::flat_hash_map<void*, TInstant>;

    explicit TStatsTable(const TStatsTableConf& conf)
        : Finished_{conf.Registry, conf.Settings.AvailablePeriod, conf.Settings.MaximumSize}
    {
    }

    void RemoveFromUnfinished(void* ctx, NTracing::TTraceId traceId) {
        Unfinished_.Write()->erase(ctx);

        NSolomon::NTracing::EraseTrace(std::move(traceId));
    }

    void RecordStart(TInstant startedAt, void* ctx) {
        (*Unfinished_.Write())[ctx] = startedAt;
    }

    void RecordFinish(TRow row, TAdditionalInfo info, void* ctx) {
        TUnfinishedTable::node_type node;
        {
            node = Unfinished_.Write()->extract(ctx);
        } // release lock ASAP to avoid holding both locks simultaneously

        if (node) {
            auto finished = Finished_.Write();
            MinimumCurrentTime_.store(finished->Insert(std::move(row), std::move(info)), std::memory_order_relaxed);
        }
    }

    TVector<TRow> GetFinishedData() const {
        return Finished_.Read()->GetData();
    }

    TVector<TRow> GetUnfinishedData() const {
        TUnfinishedTable unfinishedCopy;
        {
            unfinishedCopy = *Unfinished_.Read();
        }
        TVector<TRow> data;
        data.reserve(unfinishedCopy.size());
        for (const auto& el: unfinishedCopy) {
            data.push_back(TRow{static_cast<::NGrpc::IRequestContextBase*>(el.first), el.second});
        }
        return data;
    }

    std::optional<std::pair<TRow, TAdditionalInfo>> GetAdditionalInfo(ui64 id) {
        {
            auto finished = Finished_.Read();
            if (finished->Contains(id)) {
                return finished->GetAdditionalInfo(id);
            }
        }
        {
            auto unfinished = Unfinished_.Read();
            auto ctx = reinterpret_cast<::NGrpc::IRequestContextBase*>(id);
            if (auto it = unfinished->find(ctx); it != unfinished->end()) {
                return std::make_pair(TRow{static_cast<::NGrpc::IRequestContextBase*>(it->first), it->second},
                        TAdditionalInfo{ctx->GetRequest()->DebugString()});
            }
        }
        return std::nullopt;
    }

    ui32 GetMaximumSize() const {
        return Finished_.Read()->GetMaximumSize();
    }

    void SetMaximumSize(ui32 value) {
        if (value != GetMaximumSize()) {
            Finished_.Write()->SetMaximumSize(value);
        }
    }

    TDuration GetAvailablePeriod() const {
        return Finished_.Read()->GetAvailablePeriod();
    }

    void SetAvailablePeriod(const TDuration& value) {
        if (value != GetAvailablePeriod()) {
            Finished_.Write()->SetAvailablePeriod(value);
        }
    }

    void SetTableSettings(const TStatsTableSettings& settings) {
        TDuration available;
        ui32 maximum;
        {
            auto finished = Finished_.Read();
            available = finished->GetAvailablePeriod();
            maximum = finished->GetMaximumSize();
        }
        if (available != settings.AvailablePeriod || maximum != settings.MaximumSize) {
            auto finished = Finished_.Write();
            finished->SetAvailablePeriod(settings.AvailablePeriod);
            MinimumCurrentTime_.store(finished->SetMaximumSize(settings.MaximumSize), std::memory_order_relaxed);
        }
    }

    TStatsTableSettings GetTableSettings() const {
        return Finished_.Read()->GetTableSettings();
    }

    TDuration GetMinimumInterestingTime() const {
        return TDuration::FromValue(MinimumCurrentTime_.load(std::memory_order_relaxed));
    }

private:
    NSync::TLightRwLock<TStatsTablePriorityQueue> Finished_;
    NSync::TLightRwLock<TUnfinishedTable> Unfinished_;
    std::atomic_uint64_t MinimumCurrentTime_{0};
};

} // namespace NSolomon::NGrpc
