#include "host_resolver.h"

#include <util/generic/yexception.h>
#include <util/generic/vector.h>
#include <util/system/fs.h>
#include <util/stream/file.h>
#include <util/folder/path.h>

namespace NSolomon {
namespace {

class TCachingResolver: public IHostResolver {
public:
    explicit TCachingResolver(IHostResolverPtr delegate)
        : Delegate_(std::move(delegate))
    {
    }

    TStringBuf Scheme() const override {
        return Delegate_->Scheme();
    }

    void Resolve(TStringBuf fullName, TAddressSet* addresses) const override {
        TStringBuf scheme, namePort;
        if (!fullName.TrySplit(TStringBuf("://"), scheme, namePort) || scheme != Delegate_->Scheme()) {
            addresses->insert(TString{fullName});
            return;
        }

        TStringBuf name, portStr;
        if (!namePort.TrySplit(':', name, portStr)) {
            name = namePort;
        }

        TString cacheDir{Delegate_->Scheme()};
        TString cachePath;
        if (NFs::Exists(cacheDir) || NFs::MakeDirectory(cacheDir)) {
            cachePath = JoinFsPaths(cacheDir, name);
        }

        TAddressSet tmp;
        try {
            Delegate_->Resolve(fullName, &tmp);
            addresses->insert(tmp.begin(), tmp.end());
        } catch (...) {
            if (cachePath) {
                Cerr << "cannot resolve hosts by " << name << ", but will try to restore from cache" << Endl;
                ReadCache(cachePath, portStr, addresses);
                return;
            }

            // rethrow exception
            throw;
        }

        if (cachePath) {
            WriteCache(cachePath, tmp);
        }
    }

    void ReadCache(const TString& path, TStringBuf portStr, TAddressSet* addressses) const {
        TFile cacheFile(path, OpenExisting);
        TFileInput cache(cacheFile);
        TString address;
        while (cache.ReadLine(address) != 0) {
            if (portStr.empty()) {
                addressses->insert(address);
            } else {
                addressses->insert(address + ':' + portStr);
            }
        }
    }

    void WriteCache(const TString& path, const TAddressSet& addresses) const {
        TFile cacheFile(path, CreateAlways);
        TFileOutput cache(cacheFile);
        for (const TString& address: addresses) {
            cache << TStringBuf{address}.Before(':') << '\n';
        }
    }

private:
    IHostResolverPtr Delegate_;
};

} // namespace

IHostResolverPtr CachingResolver(IHostResolverPtr delegate) {
    return std::make_unique<TCachingResolver>(std::move(delegate));
}

} // namespace NSolomon
