#include "host_resolver.h"

#include <library/cpp/openssl/io/stream.h>
#include <library/cpp/http/io/stream.h>

#include <util/network/sock.h>

namespace NSolomon {
namespace {

const TString CONDUCTOR_HOST = "c.yandex-team.ru";

class TConductorResolver: public IHostResolver {
public:
    TConductorResolver(TStringBuf scheme, TStringBuf apiMethod)
        : Scheme_(scheme)
        , ApiMethod_(apiMethod)
    {
    }

    TStringBuf Scheme() const override {
        return Scheme_;
    }

    void Resolve(TStringBuf fullName, TAddressSet* addresses) const override {
        TStringBuf scheme, namePort;
        if (!fullName.TrySplit(TStringBuf("://"), scheme, namePort) || scheme != Scheme_) {
            addresses->insert(TString{fullName});
            return;
        }

        ui16 port = 0;
        TStringBuf name, portStr;
        if (namePort.TrySplit(':', name, portStr)) {
            TryFromString<ui16>(portStr, port);
        } else {
            name = namePort;
        }

        auto consumeHost = [addresses, port, portStr](const TString& host) {
            if (port == 0) {
                addresses->insert(host);
            } else {
                addresses->insert(host + ':' + portStr);
            }
        };

        // first try to find group in local in-memory cache, to avoid sending the same requests
        // to Conductor over and over again
        if (auto it = HostsPerName_.find(name); it != HostsPerName_.end()) {
            for (const TString& host: it->second) {
                consumeHost(host);
            }
            return;
        }

        TSocket s(TNetworkAddress(CONDUCTOR_HOST, 443), TDuration::Seconds(10));
        TSocketOutput so(s);
        TSocketInput si(s);
        TOpenSslClientIO ssl(&si, &so);

        {
            THttpOutput output(&ssl);
            output << TStringBuf("GET ") << ApiMethod_ << name << TStringBuf(" HTTP/1.1\r\n")
                   << TStringBuf("Host: ") << CONDUCTOR_HOST << TStringBuf("\r\n")
                   << TStringBuf("\r\n");
            output.Finish();
        }

        THttpInput input(&ssl);
        auto status = ParseHttpRetCode(input.FirstLine());
        if (status != 200) {
            ythrow yexception() << "cannot resolve  \'" << fullName << "\', http status: " << status;
        }

        TVector<TString> hosts;
        TString host;
        while (input.ReadLine(host)) {
            hosts.push_back(host);
            consumeHost(host);
        }
        HostsPerName_.emplace(TString{name}, std::move(hosts));
    }

private:
    TStringBuf Scheme_;
    TStringBuf ApiMethod_;
    mutable THashMap<TString, TVector<TString>> HostsPerName_;
};

} // namespace

IHostResolverPtr ConductorGroupResolver() {
    return std::make_unique<TConductorResolver>("conductor_group", "/api/groups2hosts/");
}

IHostResolverPtr ConductorTagResolver() {
    return std::make_unique<TConductorResolver>("conductor_tag", "/api/tags2hosts/");
}

} // namespace NSolomon
