#pragma once

#include <util/generic/string.h>
#include <util/generic/hash_set.h>

namespace NSolomon {

using TAddressSet = THashSet<TString>;

/**
 * Interface of host resolver.
 */
struct IHostResolver {
    virtual ~IHostResolver() = default;

    /**
     * Returns prefix (in form of URL scheme) which is identifies resolver.
     *
     * @return URL schema
     */
    virtual TStringBuf Scheme() const = 0;

    /**
     * Resolves given name into set of addresses.
     *
     * @param name        name to resolve
     * @param addresses   output set of addresses
     */
    virtual void Resolve(TStringBuf name, TAddressSet* addresses) const = 0;
};

using IHostResolverPtr = std::unique_ptr<IHostResolver>;

/**
 * Creates implementation which using file system to cache reolved addresses.
 * In case of delegat failures it will use previously cached data instead of
 * propagating error.
 *
 * @param delegate resolver which will be wrapped
 * @return caching resolver implementation
 */
IHostResolverPtr CachingResolver(IHostResolverPtr delegate);

/**
 * Creates resolver which will multiplex given resolvers.
 *
 * @param delegates resolvers which will be wrapped
 * @return multiplexing resolver implementation
 */
IHostResolverPtr MuxResolver(TVector<IHostResolverPtr> delegates);

/**
 * Creates resolver that using https://c.yandex-team.ru to convert groups to addresses.
 *
 * @return conductor resolver implementation
 */
IHostResolverPtr ConductorGroupResolver();

/**
 * Creates resolver that using https://c.yandex-team.ru to convert tags to addresses.
 *
 * @return conductor resolver implementation
 */
IHostResolverPtr ConductorTagResolver();

} // namespace NSolomon
