#include <solomon/libs/cpp/host_resolver/host_resolver.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/folder/path.h>
#include <util/stream/file.h>
#include <util/system/file.h>

using namespace NSolomon;

class TTestResolver: public IHostResolver {
public:
    explicit TTestResolver(TStringBuf scheme)
        : Scheme_{scheme}
    {
    }

    TStringBuf Scheme() const override {
        return Scheme_;
    }

    void Resolve(TStringBuf name, TAddressSet* addresses) const override {
        TStringBuf scheme, hostPort;
        Y_ENSURE(name.TrySplit(TStringBuf("://"), scheme, hostPort));
        Y_ENSURE(Scheme_ == scheme);
        addresses->insert(TString{hostPort});
    }

private:
    TStringBuf Scheme_;
};

class TFailingResolver: public IHostResolver {
public:
    explicit TFailingResolver(TStringBuf scheme)
        : Scheme_{scheme}
    {
    }

    TStringBuf Scheme() const override {
        return Scheme_;
    }

    void Resolve(TStringBuf /*name*/, TAddressSet* /*addresses*/) const override {
        ythrow yexception() << "not implemented";
    }

private:
    TStringBuf Scheme_;
};

TEST(THostResolverTest, Mux) {
    TVector<IHostResolverPtr> resolvers;
    resolvers.push_back(std::make_unique<TTestResolver>("a"));
    resolvers.push_back(std::make_unique<TTestResolver>("b"));
    resolvers.push_back(std::make_unique<TTestResolver>("c"));

    auto r = MuxResolver(std::move(resolvers));
    ASSERT_EQ(r->Scheme(), "");

    TAddressSet addresses;
    r->Resolve("a://host-a:1111", &addresses);
    r->Resolve("b://host-b:2222", &addresses);
    r->Resolve("c://host-c:3333", &addresses);

    ASSERT_EQ(addresses.size(), 3u);
    EXPECT_TRUE(addresses.contains("host-a:1111"));
    EXPECT_TRUE(addresses.contains("host-b:2222"));
    EXPECT_TRUE(addresses.contains("host-c:3333"));

    try {
        r->Resolve("d://host-d:4444", &addresses);
        FAIL() << "expected exception not thrown";
    } catch (const yexception& e) {
        ASSERT_TRUE(e.AsStrBuf().Contains("unsupported host resolver scheme: d"));
    }
}

TEST(THostResolverTest, Caching) {
    {
        auto r = CachingResolver(std::make_unique<TTestResolver>("solomon"));
        TAddressSet addresses;
        r->Resolve("solomon://some-host:1234", &addresses);

        ASSERT_EQ(addresses.size(), 1u);
        EXPECT_TRUE(addresses.contains("some-host:1234"));
    }

    TFile cacheFile(JoinFsPaths("solomon", "some-host"), OpenExisting);
    TString cache = TFileInput{cacheFile}.ReadAll();
    ASSERT_EQ("some-host\n", cache);

    {
        auto r = CachingResolver(std::make_unique<TFailingResolver>("solomon"));
        TAddressSet addresses;
        r->Resolve("solomon://some-host:1234", &addresses);

        ASSERT_EQ(addresses.size(), 1u);
        EXPECT_TRUE(addresses.contains("some-host:1234"));
    }
}

TEST(THostResolverTest, ConductorGroup) {
    auto r = ConductorGroupResolver();
    ASSERT_EQ("conductor_group", r->Scheme());

    // with port
    {
        TAddressSet addresses;
        r->Resolve("conductor_group://solomon_pre_alerting:1234", &addresses);

        ASSERT_FALSE(addresses.empty());
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-00.mon.yandex.net:1234"));
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-01.mon.yandex.net:1234"));
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-02.mon.yandex.net:1234"));
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-03.mon.yandex.net:1234"));
    }

    // without port
    {
        TAddressSet addresses;
        r->Resolve("conductor_group://solomon_pre_alerting", &addresses);

        ASSERT_FALSE(addresses.empty());
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-00.mon.yandex.net"));
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-01.mon.yandex.net"));
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-02.mon.yandex.net"));
        EXPECT_TRUE(addresses.contains("pre-alerting-vla-03.mon.yandex.net"));
    }
}

TEST(THostResolverTest, ConductorTag) {
    auto r = ConductorTagResolver();
    ASSERT_EQ("conductor_tag", r->Scheme());

    // TODO: add tags
}
