#include <solomon/libs/cpp/http/client/curl/client.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/scope.h>
#include <util/string/ascii.h>
#include <util/string/cast.h>
#include <util/string/split.h>
#include <util/string/strip.h>
#include <util/system/event.h>

using namespace NLastGetopt;
using namespace NSolomon;
using namespace NMonitoring;

void DumpHeaders(const IHeaders& headers) {
    Cerr << " <<<\n";
    headers.ForEach([] (auto name, auto value) {
        Cerr << name << ": " << value << "\n";
    });
    Cerr << " <<<\n";
}

void PrintResponse(IHttpClient::TResult result, bool verbose) {
    if (!result.Success()) {
        auto& err = result.Error();
        Cerr << "Request failed: " << (size_t)err.Type() << " " << result.Error().Message() << Endl;
        return;
    }

    auto& response = *result.Value();
    if (verbose) {
        DumpHeaders(response.Headers());
    }

    Cerr << response.Data();
}

int main(int argc, const char** argv) {
    TString methodStr;
    TVector<TString> headers;
    bool verbose{false};

    TOpts opts = TOpts::Default();
    opts.AddHelpOption();
    opts.AddLongOption('X', "method")
        .RequiredArgument("METHOD")
        .StoreResult(&methodStr)
        .DefaultValue(TString{"GET"});

    opts.AddLongOption('H', "header")
        .RequiredArgument("HEADER")
        .AppendTo(&headers);

    opts.AddCharOption('v')
        .NoArgument()
        .SetFlag(&verbose);

    opts.SetFreeArgsNum(1);
    TOptsParseResult res(&opts, argc, argv);

    const auto args = res.GetFreeArgs();
    const auto& url = args[0];

    for (auto& c: methodStr) {
        c = AsciiToUpper(c);
    }

    EHttpMethod method;
    if (!TryFromString<EHttpMethod>(methodStr, method)) {
        Cerr << "Unsupported method " << methodStr << Endl;
        return 1;
    }

    TAutoEvent done;

    auto h = Headers();
    if (!headers.empty()) {
        for (auto&& header: headers) {
            TStringBuf name, value;
            if (!TStringBuf{header}.TrySplit(":", name, value)) {
                Cerr << "Invalid header format" << Endl;
                return 1;
            }

            h->Add(StripString(name), StripString(value));
        }
    }

    TMetricRegistry r;
    IHttpClientPtr client = CreateCurlClient({}, r);
    client->Request(
        CreateRequest(method, url, {}, std::move(h)),
        [&] (auto result) {
            PrintResponse(std::move(result), verbose);
            done.Signal();
        }
    );

    done.Wait();
}
