#include "client.h"
#include "event_loop.h"

#include <library/cpp/monlib/metrics/metric_sub_registry.h>

#include <util/system/thread.h>
#include <util/thread/pool.h>

namespace NSolomon {
namespace {

class TCurlClient final: public IHttpClient {
    struct TWorker: private TNonCopyable {
        TEventLoop EventLoop;
        TThread Thread;
        std::atomic<bool> Stoped{false};

        TWorker(NMonitoring::IMetricRegistry& registry, const TCurlClientOptions& opts, ILimiterPtr limiter)
            : EventLoop{registry, opts, std::move(limiter)}
            , Thread{TThread::TParams(&TWorker::Run, this)
                    .SetName(opts.Name + ".wrk")}
        {
            Thread.Start();
        }

        ~TWorker() {
            Stoped = true;
            EventLoop.Stop();
            if (Thread.Running()) {
                Thread.Join();
            }
        }

        static void* Run(void* ctx) {
            TWorker* worker = static_cast<TWorker*>(ctx);
            while (!worker->Stoped.load(std::memory_order_relaxed)) {
                worker->EventLoop.Iteration();
            }

            // trigger callbacks for all scheduled requests
            worker->EventLoop.FinalizeAllRequests();
            return nullptr;
        }

        void HandleRequest(std::unique_ptr<TRequestContext> req) noexcept {
            EventLoop.Request(std::move(req));
        }
    };
public:
    TCurlClient(TCurlClientOptions opts, NMonitoring::TMetricRegistry& registry)
        : Options_{std::move(opts)}
    {
        if (Options_.HandlerThreads > 0) {
            HandlersPool_ = std::make_unique<TSimpleThreadPool>(TThreadPoolParams{Options_.Name + ".hndlr"});
            HandlersPool_->Start(Options_.HandlerThreads);
        }

        auto limiter = (Options_.MaxInflight != 0)
            ? CreateLimiter(Options_.MaxInflight)
            : CreateFakeLimiter();

        NMonitoring::TMetricSubRegistry subRegistry{{{"client", Options_.Name}}, &registry};
        for (auto i = 0u; i < Options_.WorkerThreads; ++i) {
            Workers_.push_back(std::make_unique<TWorker>(subRegistry, Options_, limiter));
        }
    }

    ~TCurlClient() {
        Workers_.clear();

        if (HandlersPool_) {
            HandlersPool_->Stop();
        }
    }

    void Request(IRequestPtr req, TOnComplete userCb, const TRequestOpts& opts) noexcept override {
        IHttpClient::TOnComplete cb;
        if (!HandlersPool_) {
            cb = std::move(userCb);
        } else {
            cb = [pool = HandlersPool_.get(), userCb = std::move(userCb)] (IHttpClient::TResult result) mutable {
                Y_UNUSED(pool->AddFunc([userCb = std::move(userCb), result = std::move(result)] () mutable {
                    userCb(std::move(result));
                }));
            };
        }

        auto reqCtx = std::make_unique<TRequestContext>(
            std::move(req),
            std::move(cb),
            TCurlRequestOpts{opts, Options_.CaCertDir.c_str(), Options_.DnsCacheLifetime},
            Options_.BindOptions);

        Workers_[NextIdx()]->HandleRequest(std::move(reqCtx));
    }

private:
    size_t NextIdx() {
        return CurrentIdx_.fetch_add(1) % Workers_.size();
    }

private:
    TCurlClientOptions Options_;
    std::unique_ptr<IThreadPool> HandlersPool_;
    std::vector<std::unique_ptr<TWorker>> Workers_;
    std::atomic<size_t> CurrentIdx_;
};

} // namespace

IHttpClientPtr CreateCurlClient(TCurlClientOptions opts, NMonitoring::TMetricRegistry& registry) {
    return new TCurlClient{std::move(opts), registry};
}

} // namespace NSolomon
