#include "event_loop.h"

namespace NSolomon {
namespace {

static int SocketShim(CURL* easy, curl_socket_t s, int what, void* self, void* socketp) {
    Y_UNUSED(socketp);
    static_cast<TEventLoop*>(self)->OnSocketEvent(easy, s, what);
    return CURLM_OK;
}

static int TimerShim(CURLM* handle, long timeout, void* self) {
    static_cast<TEventLoop*>(self)->OnTimerEvent(handle, timeout);
    return CURLM_OK;
}

} // namespace

TEventLoop::TEventLoop(NMonitoring::IMetricFactory& metrics, const TCurlClientOptions& options, ILimiterPtr limiter)
    : Poller_{metrics}
    , RequestQueue_{metrics, options.QueueSizeLimit}
    , PostponedQueue_{metrics}
    , CurlHandle_{metrics, options.MaxInflight}
    , HandlePool_{CreateHandlePool(metrics)}
    , Counters_{metrics}
    , Limiter_{std::move(limiter)}
{
    Poller_.AddWait(Event_.Fd(), CURL_POLL_IN, &Event_);
    CurlHandle_.SetTimerFunction(TimerShim, this);
    CurlHandle_.SetSocketFunction(SocketShim, this);
}

void TEventLoop::Iteration() noexcept {
    ProcessTasks();
    ProcessEvents();
    FinalizeFinishedRequests();
    ProcessPostponed();
}

void TEventLoop::ProcessTasks() {
    while (Limiter_->TryInc()) {
        if (auto req = RequestQueue_.Dequeue()) {
            InitRequest(std::move(req));
        } else {
            // return back taken ticket
            Limiter_->Dec();
            break;
        }
    }
}

void TEventLoop::ProcessEvents() {
    static constexpr auto MAX_POLL = 10;

    void* events[MAX_POLL];
    int filters[MAX_POLL];

    TInstant deadline = Min(Timeout_.ToDeadLine(), PostponedQueue_.NextDeadline());
    const size_t r = Poller_.WaitD(events, filters, Y_ARRAY_SIZE(events), deadline);
    bool wakeup{false};

    if (r > 0) {
        for (size_t i = 0; i < r; ++i) {
            if (&Event_ == events[i]) {
                wakeup = true;
            } else {
                CurlHandle_.SocketAction(static_cast<TRequestContext*>(events[i])->Sock(), filters[i]);
            }
        }
    } else {
        wakeup = true;
    }

    if (wakeup) {
        CurlHandle_.SocketAction(CURL_SOCKET_TIMEOUT, 0);
        Event_.Reset();
    }
}

void TEventLoop::FinalizeFinishedRequests() {
    const auto now = TInstant::Now();

    while (auto* info = CurlHandle_.ReadInfo()) {
        if (info->msg == CURLMSG_DONE) {
            auto ctx = TRequestContext::FromHandle(info->easy_handle);
            CurlHandle_.RemoveHandle(info->easy_handle);
            Poller_.Unwait(ctx->Sock());

            if (ctx->Finalize(info->data.result)) {
                Counters_.RequestDone(ctx->Time());
            } else {
                auto deadline = now + ctx->RetryBackoff();
                // TODO(ivanzhukov@): RequestQueue_'s max size can be limited -- limit PostponedQueue_ as well.
                //  Otherwise PostponedQueue_ could grow indefinitely and cause OOM
                PostponedQueue_.Push(std::move(ctx), deadline);
            }

            Limiter_->Dec();
        }
    }
}

void TEventLoop::FinalizeAllRequests() {
    TRequestError error{TRequestError::EType::RequestInitializationFailed, "Client is shutting down"};

    while (auto req = RequestQueue_.Dequeue()) {
        req->Finalize(error);
    }

    while (auto req = PostponedQueue_.Pop()) {
        req->Finalize(error);
    }

    for (auto* h: CurlHandle_.AssociatedHandles()) {
        auto ctx = TRequestContext::FromHandle(h);
        CurlHandle_.RemoveHandle(h);

        Poller_.Unwait(ctx->Sock());
        ctx->Finalize(error);
    }
}

void TEventLoop::ProcessPostponed() {
    const auto now = TInstant::Now();
    while (Limiter_->TryInc()) {
        if (PostponedQueue_.NextDeadline() <= now) {
            InitRequest(PostponedQueue_.Pop());
        } else {
            Limiter_->Dec();
            break;
        }
    }
}

void TEventLoop::InitRequest(std::unique_ptr<TRequestContext> ctx) noexcept {
    const auto t = ctx->Init(HandlePool_->GetHandle(), Share_.Get());
    if (CurlHandle_.AddHandle(ctx->Handle())) {
        Counters_.RequestStarted(t);

        // ownership of this request context has been transferred to curl handle
        (void) ctx.release();
    } else {
        ctx->Finalize(
            TRequestError{TRequestError::EType::RequestInitializationFailed, "Internal error"}
        );
        Poller_.Unwait(ctx->Sock());
        Limiter_->Dec();
    }
}

void TEventLoop::OnSocketEvent(CURL* easy, curl_socket_t s, int what) {
    auto ctx = TRequestContext::FromHandle(easy);
    ctx->SetSocket(s);
    Poller_.AddWait(s, what, ctx.release());
}

void TEventLoop::OnTimerEvent(CURLM* handle, long timeoutMillis) {
    Y_UNUSED(handle);
    if (timeoutMillis <= 0) {
        Event_.Signal();
        return;
    }

    Timeout_ = TDuration::MilliSeconds(timeoutMillis);
}

} // namespace NSolomon
