#pragma once

#include "poller.h"
#include "request_queue.h"
#include "handle_pool.h"
#include "multi_handle.h"
#include "postponed_queue.h"
#include "request_context.h"
#include "share.h"

#include <solomon/libs/cpp/event/event.h>

namespace NSolomon {

class TEventLoop {
    struct TCounters {
        explicit TCounters(NMonitoring::IMetricFactory& metrics) {
            Done = metrics.Rate(NMonitoring::MakeLabels({{"sensor", "httpClient.requestsDone"}}));
            Timings = metrics.HistogramRate(
                NMonitoring::MakeLabels({{"sensor", "httpClient.requestTimeMillis"}}),
                NMonitoring::ExponentialHistogram(13, 2, 16));
            WaitTimes = metrics.HistogramRate(
                NMonitoring::MakeLabels({{"sensor", "httpClient.requestWaitTimeMillis"}}),
                NMonitoring::ExponentialHistogram(13, 2, 16));
        }

        void RequestDone(TDuration t) const {
            Done->Inc();
            Timings->Record(t.MilliSeconds());
        }

        void RequestStarted(TDuration t) const {
            WaitTimes->Record(t.MilliSeconds());
        }

        NMonitoring::IRate* Done{nullptr};
        NMonitoring::IHistogram* Timings{nullptr};
        NMonitoring::IHistogram* WaitTimes{nullptr};
    };

public:
    TEventLoop(NMonitoring::IMetricFactory& metrics, const TCurlClientOptions& options, ILimiterPtr limiter);

    ~TEventLoop() {
        Stop();
    }

    void Stop() noexcept {
        Event_.Signal();
    }

    void Request(std::unique_ptr<TRequestContext> req) noexcept {
        req = RequestQueue_.TryEnqueue(std::move(req));
        if (Y_UNLIKELY(req)) {
            auto deadline = TInstant::Now() + req->RetryBackoff();
            PostponedQueue_.Push(std::move(req), deadline);
        }

        Event_.Signal();
    }

    void Iteration() noexcept;

    void FinalizeAllRequests();

    void OnSocketEvent(CURL* easy, curl_socket_t s, int what);

    void OnTimerEvent(CURLM* handle, long timeoutMillis);

private:
    void ProcessTasks();

    void ProcessEvents();

    void FinalizeFinishedRequests();

    void ProcessPostponed();

    void InitRequest(std::unique_ptr<TRequestContext> ctx) noexcept;

private:
    TDuration Timeout_{TDuration::Max()};
    TCurlSocketPoller Poller_;
    TPollableEvent Event_;
    TRequestQueue<TRequestContext> RequestQueue_;
    TPostponedQueue<TRequestContext> PostponedQueue_;
    TCurlMultiHandle CurlHandle_;
    TCurlShare Share_;
    std::unique_ptr<ICurlHandlePool> HandlePool_;
    TCounters Counters_;
    ILimiterPtr Limiter_;
};

} // namespace NSolomon
