#include "handle_pool.h"

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/deque.h>

namespace NSolomon {
namespace {

using namespace NMonitoring;

class THandlePool final: public ICurlHandlePool {
    struct TCounters {
        explicit TCounters(IMetricFactory& metrics) {
            ActiveHandles = metrics.IntGauge(MakeLabels({{"sensor", "httpClient.handlePool.size"}}));
            Spawned = metrics.Rate(MakeLabels({{"sensor", "httpClient.handlePool.spawned"}}));
            Returned = metrics.Rate(MakeLabels({{"sensor", "httpClient.handlePool.returned"}}));
            InUse = metrics.IntGauge(MakeLabels({{"sensor", "httpClient.handlePool.inUse"}}));
        }

        void HandleSpawned() const {
            ActiveHandles->Inc();
            Spawned->Inc();
            InUse->Inc();
        }

        void HandleTaken() const {
            InUse->Inc();
        }

        void HandleReturned() const {
            Returned->Inc();
            InUse->Dec();
        }

        IIntGauge* ActiveHandles{nullptr};
        IIntGauge* InUse{nullptr};
        IRate* Spawned{nullptr};
        IRate* Returned{nullptr};
    };

    class THandleImpl final: public ICurlHandle {
    public:
        THandleImpl(CURL* handle, THandlePool* pool)
            : ICurlHandle{handle}
            , Pool_{pool}
        {
        }

        ~THandleImpl() override {
            Pool_->ReturnHandle(Handle());
        }

    private:
        THandlePool* Pool_{nullptr};
    };

public:
    explicit THandlePool(NMonitoring::IMetricFactory& metrics)
        : Counters_{metrics}
    {
    }

    ~THandlePool() override {
        for (auto* h: KnownHandles_) {
            curl_easy_cleanup(h);
        }
    }

    void ReturnHandle(CURL* handle) {
        Handles_.push_back(handle);
        Counters_.HandleReturned();
    }

    TStats Stats() const override {
        return {
            .TotalSize = KnownHandles_.size(),
            .HandlesAvailable = Handles_.size(),
            .HandlesUsed = KnownHandles_.size() - Handles_.size(),
        };
    }

    std::unique_ptr<ICurlHandle> GetHandle() override {
        CURL* handle{nullptr};
        if (Handles_.empty()) {
            handle = curl_easy_init();
            KnownHandles_.push_back(handle);
            Counters_.HandleSpawned();
        } else {
            handle = Handles_.front();
            Handles_.pop_front();
            Counters_.HandleTaken();
        }

        return std::make_unique<THandleImpl>(handle, this);
    }

private:
    TDeque<CURL*> Handles_;

    // all the handles created in this pool to clean them up on destruction
    TVector<CURL*> KnownHandles_;
    TCounters Counters_;
};

} // namespace

std::unique_ptr<ICurlHandlePool> CreateHandlePool(NMonitoring::IMetricFactory& metrics) {
    return std::make_unique<THandlePool>(metrics);
}

} // namespace NSolomon
