#pragma once

#include <solomon/libs/cpp/http/client/http.h>

#include <util/generic/size_literals.h>
#include <util/string/ascii.h>

#include <curl/curl.h>

namespace NSolomon {

struct THeaders final: public IHeaders {
    TMaybe<TStringBuf> Find(TStringBuf name) const override {
        const auto it = FindIf(Headers_, [&] (auto&& header) {
            return AsciiEqualsIgnoreCase(header.first, name);
        });

        return it == Headers_.end() ? Nothing() : TMaybe<TStringBuf>(it->second);
    }

    void Add(TStringBuf name, TStringBuf value) override {
        Headers_.emplace_back(TString{name}, TString{value});
    }

    void ForEach(std::function<void(TStringBuf, TStringBuf)> fn) const override {
        for (auto&& [name, value]: Headers_) {
            fn(name, value);
        }
    }

private:
    TVector<std::pair<TString, TString>> Headers_;
};

class THeaderArena {
public:
    THeaderArena() {
        Data_.reserve(1_KB);
    }

    ~THeaderArena() {
        curl_slist_free_all(List_);
    }

    bool Empty() {
        return Data_.empty();
    }

    void Add(TStringBuf name, TStringBuf value) {
        static constexpr TStringBuf DELIM = ": ";

        auto headerOffset = Data_.size();

        Copy(name.begin(), name.end(), std::back_inserter(Data_));
        Copy(DELIM.begin(), DELIM.end(), std::back_inserter(Data_));
        Copy(value.begin(), value.end(), std::back_inserter(Data_));
        Data_.push_back('\0');

        List_ = curl_slist_append(List_, Data_.data() + headerOffset);
    }

    void Bind(CURL* h) {
        curl_easy_setopt(h, CURLOPT_HTTPHEADER, List_);
    }

private:
    TVector<char> Data_;
    struct curl_slist* List_{nullptr};
};

} // namespace NSolomon
