#pragma once

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/generic/hash_set.h>

#include <curl/curl.h>

namespace NSolomon {

class TCurlMultiHandle {
public:
    struct TCounters {
        TCounters(NMonitoring::IMetricFactory& metrics, ui64 maxInflight) {
            InFlight = metrics.IntGauge(NMonitoring::MakeLabels({{"sensor", "httpClient.requestsInFlight"}}));
            InFlightLimit = metrics.IntGauge(NMonitoring::MakeLabels({{"sensor", "httpClient.requestsInFlightLimit"}}));
            InFlightLimit->Set(maxInflight);
        }

        void SetInflight(size_t val) const {
            InFlight->Set(val);
        }

        NMonitoring::IIntGauge* InFlight;
        NMonitoring::IIntGauge* InFlightLimit;
    };

    using TSocketFunction = int (CURL*, curl_socket_t, int, void*, void*);
    using TTimerFunction = int (CURLM*, long, void*);

    TCurlMultiHandle(NMonitoring::IMetricFactory& metrics, ui64 maxInflight)
        : Counters_{metrics, maxInflight}
        , Handle_{curl_multi_init()}
    {
    }

    ~TCurlMultiHandle() {
        curl_multi_cleanup(Handle_);
    }

    bool AddHandle(CURL* handle) {
        const auto ret = curl_multi_add_handle(Handle_, handle);
        Y_VERIFY(ret == CURLM_OK);
        auto [_, ok] = EasyHandles_.emplace(handle);
        Y_VERIFY_DEBUG(ok);
        return ret == CURLM_OK;
    }

    void RemoveHandle(CURL* handle) {
        const auto ret = curl_multi_remove_handle(Handle_, handle);
        Y_VERIFY(ret == CURLM_OK);
        const auto erased = EasyHandles_.erase(handle);
        Y_VERIFY_DEBUG(erased == 1);
    }

    THashSet<CURL*> AssociatedHandles() const {
        return EasyHandles_;
    }

    void SetSocketFunction(TSocketFunction fn, void* data) {
        CurlMultiSet(Handle_, CURLMOPT_SOCKETFUNCTION, fn);
        CurlMultiSet(Handle_, CURLMOPT_SOCKETDATA, data);
    }

    void SetTimerFunction(TTimerFunction fn, void* data) {
        CurlMultiSet(Handle_, CURLMOPT_TIMERFUNCTION, fn);
        CurlMultiSet(Handle_, CURLMOPT_TIMERDATA, data);
    }

    void SocketAction(curl_socket_t sockfd, int evBitmask) {
        const auto ret = curl_multi_socket_action(Handle_, sockfd, evBitmask, &InFlight_);
        Y_VERIFY(ret == CURLM_OK);
        Counters_.SetInflight(InFlight_);
    }

    CURLMsg* ReadInfo() {
        int msgsInQueue;
        return curl_multi_info_read(Handle_, &msgsInQueue);
    }

    int InFlight() const {
        return InFlight_;
    }

private:
    template <typename... TArgs>
    void CurlMultiSet(CURLM* curl, CURLMoption opt, TArgs&&... args) {
        const auto ret = curl_multi_setopt(curl, opt, std::forward<TArgs>(args)...);
        Y_VERIFY(ret == CURLM_OK);
    }

private:
    int InFlight_{0};
    TCounters Counters_;
    CURLM* Handle_;
    THashSet<CURL*> EasyHandles_;
};

} // namespace NSolomon
