#include "poller.h"

#include <library/cpp/monlib/metrics/metric_registry.h>

#include <util/network/pollerimpl.h>
#include <util/generic/hash_set.h>

using namespace NMonitoring;

namespace NSolomon {
    struct TCurlSocketPoller::TImpl: public TPollerImpl<TWithoutLocking> {
        using TBase = TPollerImpl<TWithoutLocking>;

        NMonitoring::IIntGauge* UsedFds;
        THashSet<curl_socket_t> Sockets;

        explicit TImpl(NMonitoring::IMetricFactory& metrics) {
            UsedFds = metrics.IntGauge(MakeLabels({{"sensor", "httpClient.poller.usedFds"}}));
        }

        void AddSocket(curl_socket_t sock, void* cookie, int what) {
            Set(cookie, sock, what);
            if (Sockets.insert(sock).second) {
                UsedFds->Set(Sockets.size());
            }
        }

        void RemoveSocket(curl_socket_t sock) {
            Remove(sock);
            if (Sockets.erase(sock) == 1) {
                UsedFds->Set(Sockets.size());
            }
        }

        inline size_t DoWaitReal(void** ev, int* filters, TEvent* events, size_t len, TInstant deadline) {
            const size_t ret = WaitD(events, len, deadline);

            for (size_t i = 0; i < ret; ++i) {
                ev[i] = ExtractEvent(&events[i]);
                filters[i] = ExtractFilter(&events[i]);
            }

            return ret;
        }

        inline size_t DoWait(void** ev, int* filters, size_t len, TInstant deadline) {
            if (len == 1) {
                TEvent tmp;
                return DoWaitReal(ev, filters, &tmp, 1, deadline);
            } else {
                TTempArray<TEvent> tmpEvents(len);
                return DoWaitReal(ev, filters, tmpEvents.Data(), len, deadline);
            }
        }

        static int ExtractFilter(const TEvent* event) noexcept {
            if (TBase::ExtractStatus(event)) {
                return CURL_CSELECT_ERR;
            }

            auto filter  = TBase::ExtractFilterImpl(event);

            int code{0};
            if (filter & CONT_POLL_READ) {
                code |= CURL_CSELECT_IN;
            }

            if (filter & CONT_POLL_WRITE) {
                code |= CURL_CSELECT_OUT;
            }

            return code;
        }
    };

    TCurlSocketPoller::TCurlSocketPoller(IMetricFactory& metrics)
        : Impl_{new TCurlSocketPoller::TImpl{metrics}}
    {
    }

    TCurlSocketPoller::~TCurlSocketPoller() = default;

    void TCurlSocketPoller::AddWait(curl_socket_t sock, int what, void* cookie) {
        switch (what) {
            case CURL_POLL_IN:
                Impl_->AddSocket(sock, cookie, CONT_POLL_READ);
                break;
            case CURL_POLL_OUT:
                Impl_->AddSocket(sock, cookie, CONT_POLL_WRITE);
                break;
            case CURL_POLL_INOUT:
                Impl_->AddSocket(sock, cookie, CONT_POLL_READ | CONT_POLL_WRITE);
                break;
            case CURL_POLL_REMOVE:
                Impl_->RemoveSocket(sock);
                break;
        }
    }

    void TCurlSocketPoller::Unwait(curl_socket_t sock) {
        if (sock != CURL_SOCKET_BAD) {
            Impl_->RemoveSocket(sock);
        }
    }

    size_t TCurlSocketPoller::WaitD(void** events, int* filters, size_t len, TInstant deadline) {
        return Impl_->DoWait(events, filters, len, deadline);
    }

} // namespace NSolomon
