#pragma once

#include "queue_metrics.h"

#include <util/datetime/base.h>
#include <util/generic/queue.h>
#include <util/system/mutex.h>

#include <mutex>

namespace NSolomon {

template <typename T>
class TPostponedQueue {
    struct TItem: private TMoveOnly {
        TItem(std::unique_ptr<T> value, TInstant deadline) noexcept
            : Value{std::move(value)}
            , Deadline{deadline}
        {
        }

        bool operator>(const TItem& other) const noexcept {
            return Deadline > other.Deadline;
        }

        mutable std::unique_ptr<T> Value;
        TInstant Deadline{TInstant::Max()};
    };

public:
    explicit TPostponedQueue(NMonitoring::IMetricFactory& metrics)
        : Metrics_{metrics, "postponedQueue", 0}
    {
    }

    TInstant NextDeadline() const noexcept {
        std::lock_guard lg{Mutex_};
        if (Queue_.empty()) {
            return TInstant::Max();
        }
        return Queue_.top().Deadline;
    }

    void Push(std::unique_ptr<T> value, TInstant deadline) {
        std::lock_guard lg{Mutex_};
        Queue_.emplace(std::move(value), deadline);
        Metrics_.Enqueue();
    }

    std::unique_ptr<T> Pop() {
        std::lock_guard lg{Mutex_};
        if (Queue_.empty()) {
            return nullptr;
        }

        auto value = std::move(Queue_.top().Value);
        Queue_.pop();
        Metrics_.Dequeue();
        return value;
    }

private:
    TQueueMetrics Metrics_;
    TPriorityQueue<TItem, TVector<TItem>, TGreater<TItem>> Queue_;
    mutable TMutex Mutex_;
};

} // namespace NSolomon
