#pragma once

#include "client.h"
#include "errors.h"
#include "handle.h"
#include "request_options.h"
#include "response.h"

#include <solomon/libs/cpp/backoff/backoff.h>
#include <solomon/libs/cpp/backoff/jitter.h>

#include <util/string/builder.h>
#include <util/string/split.h>
#include <util/string/strip.h>
#include <util/system/hp_timer.h>

#include <curl/curl.h>

namespace NSolomon {

class TRequestContext final: TMoveOnly {
public:
    TRequestContext(
        IRequestPtr req,
        IHttpClient::TOnComplete callback,
        TCurlRequestOpts opts,
        std::optional<TCurlBindOptions> bindOptions)
        : Request_{std::move(req)}
        , Callback_{std::move(callback)}
        , Opts_{opts}
        , Url_{TString{Request_->Url()}}
        , UploadDataView_{Request_->Data()}
        , CaPath_{opts.CaPath}
        , DnsCacheLifetime_{opts.DnsCacheLifetime}
        , Retries_{opts.Retries}
        , Backoff_{opts.BackoffMin, opts.BackoffMax}
        , BindOptions_{std::move(bindOptions)}
    {
    }

    static std::unique_ptr<TRequestContext> FromHandle(CURL* handle) {
        TRequestContext* ctx;
        CURLcode ret = curl_easy_getinfo(handle, CURLINFO_PRIVATE, &ctx);
        Y_VERIFY(ret == CURLE_OK);
        return std::unique_ptr<TRequestContext>(ctx);
    }

    void Finalize(TRequestError error) noexcept;

    bool Finalize(CURLcode result) noexcept;

    TDuration Init(std::unique_ptr<ICurlHandle> handle, CURLSH* share);

    CURL* Handle() const {
        return Handle_->Handle();
    }

    curl_socket_t Sock() const {
        return Socket_;
    }

    void SetSocket(curl_socket_t s) {
        Socket_ = s;
    }

    TDuration Time() const {
        return TDuration::Seconds(Timer_.Passed());
    }

    TDuration RetryBackoff() {
        return Backoff_();
    }

    int OnDebug(CURL* handle, curl_infotype type, TStringBuf msg);

    size_t OnHeader(TStringBuf data);

    size_t OnBody(TStringBuf data);

    int OnSeek(curl_off_t offset, int origin);

    int OnRead(char* buffer, size_t capacity);

    curl_socket_t OnSocketOpen(curlsocktype, struct curl_sockaddr* address);

private:
    void EnsureResponse() {
        if (!Response_) {
            Response_.Reset(new TResponse);
        }
    }

    void WriteHeaders(CURL* h);

    ui32 Retries() const {
        return Retries_;
    }

private:
    bool HasResponseLimit() const {
        return Opts_.ResponseLimitBytes != 0;
    }

    bool FitsResponseLimit(TStringBuf strVal) const {
        ui64 val{};
        if (!TryFromString(strVal, val)) {
            return false;
        }

        return FitsResponseLimit(val);
    }

    bool FitsResponseLimit(ui64 val) const {
        return val < Opts_.ResponseLimitBytes;
    }

    template <typename T>
    void ResponseLimitSizeExceeded(T val) {
        Error_ = {TRequestError::EType::ResponseTooLarge,
                  TStringBuilder() << "Response size limit is " << Opts_.ResponseLimitBytes << " bytes, but response is " << val << " bytes"};
    }

    TRequestError GetError(CURLcode code) const {
        if (Error_) {
            return *Error_;
        } else if (code == CURLE_OPERATION_TIMEDOUT && Socket_ == CURL_SOCKET_BAD) {
            return TRequestError{TRequestError::EType::ConnectFailed, TStringBuilder() << "Connection to " << Url_ << " failed: " << ErrorMessage_};
        }

        return TRequestError{FromCurlError(code), ErrorMessage_};
    }

private:
    THPTimer Timer_;
    IRequestPtr Request_;
    IHttpClient::TOnComplete Callback_;
    TCurlRequestOpts Opts_;
    TString Url_;
    THeaderArena HeaderArena_;
    TStringBuf UploadDataView_;
    char ErrorMessage_[CURL_ERROR_SIZE];
    THolder<TResponse> Response_;
    std::unique_ptr<ICurlHandle> Handle_;
    curl_socket_t Socket_{CURL_SOCKET_BAD};
    const char* CaPath_;
    TDuration DnsCacheLifetime_;
    ui32 Retries_{0};
    TExpBackoff<THalfJitter> Backoff_;
    std::optional<TRequestError> Error_;
    std::optional<TCurlBindOptions> BindOptions_;
};

} // namespace NSolomon
