#pragma once

#include "queue_metrics.h"

#include <solomon/libs/cpp/limiter/limiter.h>

#include <util/thread/lfqueue.h>

namespace NSolomon {

template <typename T>
class TRequestQueue {
public:
    TRequestQueue(NMonitoring::IMetricFactory& metrics, size_t sizeLimit)
        : Metrics_{metrics, "requestQueue", sizeLimit}
        , SizeLimiter_{sizeLimit ? CreateLimiter(sizeLimit) : CreateFakeLimiter()}
    {
    }

    virtual ~TRequestQueue() {
        T* item = nullptr;
        while (Queue_.Dequeue(&item)) {
            delete item;
        }
    }

    [[nodiscard]]
    std::unique_ptr<T> TryEnqueue(std::unique_ptr<T> req) {
        if (!SizeLimiter_->TryInc()) {
            Metrics_.Drop();
            return req; // return it back if we hit the limit
        }

        Queue_.Enqueue(req.release());
        Metrics_.Enqueue();

        return nullptr;
    }

    [[nodiscard]]
    std::unique_ptr<T> Dequeue() {
        T* item = nullptr;
        if (Queue_.Dequeue(&item)) {
            SizeLimiter_->Dec();
            Metrics_.Dequeue();
            return std::unique_ptr<T>(item);
        }

        return nullptr;
    }

    bool IsEmpty() {
        return Queue_.IsEmpty();
    }

private:
    TQueueMetrics Metrics_;
    ILimiterPtr SizeLimiter_;
    TLockFreeQueue<T*> Queue_;
};

} // namespace NSolomon
