#pragma once

#include "method.h"
#include "request_error.h"

#include <solomon/libs/cpp/error_or/error_or.h>

#include <library/cpp/http/misc/httpcodes.h>
#include <library/cpp/monlib/metrics/fwd.h>

#include <util/datetime/base.h>
#include <util/generic/maybe.h>
#include <util/generic/vector.h>
#include <util/network/address.h>

namespace NSolomon {
    struct IHeaders {
        virtual ~IHeaders() = default;
        virtual TMaybe<TStringBuf> Find(TStringBuf) const = 0;
        virtual void Add(TStringBuf, TStringBuf) = 0;
        virtual void ForEach(std::function<void(TStringBuf, TStringBuf)>) const = 0;
    };

    using IHeadersPtr = THolder<IHeaders>;

    struct IRequest {
        virtual ~IRequest() = default;

        virtual EHttpMethod Method() const = 0;
        virtual TStringBuf Url() const = 0;
        virtual TStringBuf Data() const = 0;

        virtual const IHeaders& Headers() const = 0;
        virtual IHeaders& Headers() = 0;
    };

    using IRequestPtr = THolder<IRequest>;

    struct IResponse {
        virtual ~IResponse() = default;
        virtual HttpCodes Code() const = 0;
        virtual TStringBuf Data() const = 0;

        // After this call response may or may not contain original data
        virtual TString ExtractData() = 0;
        virtual const IHeaders& Headers() const = 0;
    };

    using IResponsePtr = THolder<IResponse>;

    struct TKeepAliveOptions {
        bool TcpKeepAlive{false};
        // Time between keep-alive probes
        TDuration KeepInterval{TDuration::Seconds(60)};
        // Keep-alive idle time
        TDuration KeepIdle{TDuration::Seconds(120)};
    };

    struct TRequestOpts {
        TDuration ConnectTimeout{TDuration::Zero()};
        TDuration ReadTimeout{TDuration::Zero()};

        ui8 Retries{0};
        TDuration BackoffMin{TDuration::Seconds(1)};
        TDuration BackoffMax{TDuration::Seconds(20)};

        /**
         * @brief Read up to this many bytes. If server response exceeds this limit, callback with error will be invoked.
         */
        ui64 ResponseLimitBytes{0};

        // NB: may not be supported on some backends/OS
        std::optional<TKeepAliveOptions> KeepAliveOpts;
    };

    /**
     * A common interface for HTTP clients
     */
    struct IHttpClient: TNonCopyable, TThrRefBase {
        using TResult = TErrorOr<IResponsePtr, TRequestError>;
        using TOnComplete = std::function<void(TResult)>;

        /**
         * Perform an asynchronous request
         */
        virtual void Request(IRequestPtr req, TOnComplete cb, const TRequestOpts& opts = {}) noexcept = 0;
    };

    using IHttpClientPtr = TIntrusivePtr<IHttpClient>;

    IHeadersPtr Headers(const TVector<std::pair<TString, TString>>& headers);
    IHeadersPtr Headers();

    IRequestPtr CreateRequest(EHttpMethod method, TString url, TString data = {}, IHeadersPtr headers = nullptr);
    IRequestPtr Get(TString url, IHeadersPtr headers = nullptr);
    IRequestPtr Post(TString url, TString data = {}, IHeadersPtr headers = nullptr);
    IRequestPtr Put(TString url, TString data = {}, IHeadersPtr headers = nullptr);

} // namespace NSolomon
